package roster

import (
	"context"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PatchV1FeaturedChannel", func() {
	var (
		server *ghttp.Server
		client Client
		err    error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	It("fails with invalid parameters", func() {
		params := v1.PatchFeaturedChannelRequestParams{
			TeamID:    "%qq",
			ChannelID: "channel-id",
		}
		err := client.PatchV1FeaturedChannel(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("invalid"))

	})

	It("calls out to the roster URL", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("PATCH", "/v1/teams/team-id/featured_channels/channel-id"),
				ghttp.VerifyJSON(`{ "display_position": 5}`),
			),
		)

		displayPosition := uint(5)
		params := v1.PatchFeaturedChannelRequestParams{
			TeamID:          "team-id",
			ChannelID:       "channel-id",
			DisplayPosition: &displayPosition,
		}

		_ = client.PatchV1FeaturedChannel(context.Background(), params, nil)

		receivedRequests := server.ReceivedRequests()
		Expect(receivedRequests).To(HaveLen(1))
	})

	It("returns nil on request roundtrip success", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("PATCH", "/v1/teams/team-id/featured_channels/channel-id"),
				ghttp.VerifyJSON(`{ "display_position": 5}`),
				ghttp.RespondWith(http.StatusNoContent, ""),
			),
		)
		displayPosition := uint(5)
		params := v1.PatchFeaturedChannelRequestParams{
			TeamID:          "team-id",
			ChannelID:       "channel-id",
			DisplayPosition: &displayPosition,
		}
		err := client.PatchV1FeaturedChannel(context.Background(), params, nil)

		Expect(err).NotTo(HaveOccurred())
	})

	It("returns error on failed update", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("PATCH", "/v1/teams/team-id/featured_channels/channel-id"),
				ghttp.RespondWith(http.StatusInternalServerError, "Surprise Error"),
			),
		)

		params := v1.PatchFeaturedChannelRequestParams{
			TeamID:    "team-id",
			ChannelID: "channel-id",
		}
		err := client.PatchV1FeaturedChannel(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("Surprise Error"))
	})

	It("returns error on resource not found", func() {
		server.AppendHandlers(
			ghttp.CombineHandlers(
				ghttp.VerifyRequest("PATCH", "/v1/teams/team-id/featured_channels/channel-id"),
				ghttp.RespondWith(http.StatusNotFound, "team not found"),
			),
		)

		params := v1.PatchFeaturedChannelRequestParams{
			TeamID:    "team-id",
			ChannelID: "channel-id",
		}
		err := client.PatchV1FeaturedChannel(context.Background(), params, nil)

		Expect(err).To(HaveOccurred())
		Expect(err.Error()).To(ContainSubstring("team not found"))
	})
})
