package roster

import (
	"context"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PatchV1Team", func() {
	var (
		server *ghttp.Server
		client Client
		err    error

		teamID                              string
		displayName, userID, description    string
		logoID, bannerID, backgroundImageID string

		patchParams v1.PatchTeamRequestParams
	)

	BeforeEach(func() {
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())

		teamID = "ID"
		displayName = "DISPLAY NAME"
		userID = "999999999"
		description = "DESCRIPTION"
	})

	AfterEach(func() {
		server.Close()
	})

	Context("with image IDs not present", func() {
		BeforeEach(func() {
			patchParams = v1.PatchTeamRequestParams{
				TeamID:              teamID,
				DisplayName:         &displayName,
				UserID:              &userID,
				DescriptionMarkdown: &description,
				LogoID:              nil,
				BannerID:            nil,
				BackgroundImageID:   nil,
			}
		})

		It("makes a PATCH request with JSON image ID fields not present", func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("PATCH", "/v1/teams/ID"),

					ghttp.VerifyJSON(`{
						"display_name": "DISPLAY NAME",
						"user_id": "999999999",
						"description_markdown": "DESCRIPTION"
					}`),
				),
			)

			_, _ = client.PatchV1Team(context.Background(), patchParams, nil)
			Expect(server.ReceivedRequests()).To(HaveLen(1))
		})
	})

	Context("with image IDs set to null", func() {
		BeforeEach(func() {
			patchParams = v1.PatchTeamRequestParams{
				TeamID:              teamID,
				DisplayName:         &displayName,
				UserID:              &userID,
				DescriptionMarkdown: &description,
				LogoID: &v1.NullString{
					Value:   nil,
					Present: true,
				},
				BannerID: &v1.NullString{
					Value:   nil,
					Present: true,
				},
				BackgroundImageID: &v1.NullString{
					Value:   nil,
					Present: true,
				},
			}
		})

		It("makes a PATCH request with JSON image ID fields set to null", func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("PATCH", "/v1/teams/ID"),

					ghttp.VerifyJSON(`{
						"display_name": "DISPLAY NAME",
						"user_id": "999999999",
						"description_markdown": "DESCRIPTION",
						"logo_id": null,
						"banner_id": null,
						"background_image_id": null
					}`),
				),
			)

			_, _ = client.PatchV1Team(context.Background(), patchParams, nil)
			Expect(server.ReceivedRequests()).To(HaveLen(1))
		})
	})

	Context("with image IDs set to values", func() {
		BeforeEach(func() {
			logoID = "LOGO"
			bannerID = "BANNER"
			backgroundImageID = "BACKGROUND"

			patchParams = v1.PatchTeamRequestParams{
				TeamID:              teamID,
				DisplayName:         &displayName,
				UserID:              &userID,
				DescriptionMarkdown: &description,
				LogoID: &v1.NullString{
					Value:   &logoID,
					Present: true,
				},
				BannerID: &v1.NullString{
					Value:   &bannerID,
					Present: true,
				},
				BackgroundImageID: &v1.NullString{
					Value:   &backgroundImageID,
					Present: true,
				},
			}
		})

		It("makes a PATCH request with JSON image ID fields set to string values", func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("PATCH", "/v1/teams/ID"),

					ghttp.VerifyJSON(`{
						"display_name": "DISPLAY NAME",
						"user_id": "999999999",
						"description_markdown": "DESCRIPTION",
						"logo_id": "LOGO",
						"banner_id": "BANNER",
						"background_image_id": "BACKGROUND"
					}`),
				),
			)

			_, _ = client.PatchV1Team(context.Background(), patchParams, nil)
			Expect(server.ReceivedRequests()).To(HaveLen(1))
		})

		It("returns error on failed update", func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest("PATCH", "/v1/teams/ID"),
					ghttp.RespondWith(http.StatusInternalServerError, "Surprise Error"),
				),
			)

			resp, err := client.PatchV1Team(context.Background(), patchParams, nil)
			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("Surprise Error"))
			Expect(resp).To(BeNil())
		})

		Context("when Roster returns the updated team", func() {
			var (
				teamName            string
				descriptionHTML     string
				descriptionMarkdown string
				logoURL             string
				bannerURL           string
				backgroundImageURL  string
			)

			BeforeEach(func() {
				teamName = "staff"
				descriptionHTML = "<strong>stuff</strong>"
				descriptionMarkdown = "**stuff**"
				logoURL = "LOGO URL"
				bannerURL = "BANNER URL"
				backgroundImageURL = "BACKGROUND IMAGE URL"

				payloadFmt := `
					{
						"data": {
							"id": "%s",
							"name": "%s",
							"display_name": "%s",
							"user_id": "%s",
							"description_html": "%s",
							"description_markdown": "%s",
							"logo_id": "%s",
							"logo_url": "%s",
							"banner_id": "%s",
							"banner_url": "%s",
							"background_image_id": "%s",
							"background_image_url": "%s"
						}
					}
					`

				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest("PATCH", "/v1/teams/ID"),

						ghttp.VerifyJSON(`{
								"display_name": "DISPLAY NAME",
								"user_id": "999999999",
								"description_markdown": "DESCRIPTION",
								"logo_id": "LOGO",
								"banner_id": "BANNER",
								"background_image_id": "BACKGROUND"
							}`),

						ghttp.RespondWith(http.StatusOK, fmt.Sprintf(
							payloadFmt,
							teamID,
							teamName,
							displayName,
							userID,
							descriptionHTML,
							descriptionMarkdown,
							logoID,
							logoURL,
							bannerID,
							bannerURL,
							backgroundImageID,
							backgroundImageURL,
						)),
					),
				)
			})

			It("returns the updated team on request roundtrip success", func() {
				resp, err := client.PatchV1Team(context.Background(), patchParams, nil)

				Expect(err).NotTo(HaveOccurred())
				Expect(resp).NotTo(BeNil())
				Expect(resp.Data.DisplayName).To(Equal(displayName))
				Expect(resp.Data.UserID).To(Equal(userID))
				Expect(resp.Data.DescriptionHTML).To(Equal(descriptionHTML))
				Expect(resp.Data.DescriptionMarkdown).To(Equal(descriptionMarkdown))
				Expect(*resp.Data.LogoID).To(Equal(logoID))
				Expect(*resp.Data.LogoURL).To(Equal(logoURL))
				Expect(*resp.Data.BannerID).To(Equal(bannerID))
				Expect(*resp.Data.BannerURL).To(Equal(bannerURL))
				Expect(*resp.Data.BackgroundImageID).To(Equal(backgroundImageID))
				Expect(*resp.Data.BackgroundImageURL).To(Equal(backgroundImageURL))
			})
		})
	})
})
