package roster

import (
	"context"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PostV1ImageUploads", func() {
	var (
		server        *ghttp.Server
		client        Client
		teamID        string
		category      string
		size          uint64
		fileType      string
		requestParams v1.PostImageUploadsRequestParams
		imageID       string
		url           string
		err           error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()

		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})

		teamID = "123"
		category = "category"
		size = 999
		fileType = "file type"

		requestParams = v1.PostImageUploadsRequestParams{
			TeamID:   teamID,
			Category: category,
			Size:     size,
			FileType: fileType,
		}

		imageID = "some-image-id"
		url = "www.upload.url.com.net"

		Expect(err).NotTo((HaveOccurred()))
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/teams/123/image_uploads"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1ImageUploads(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a 404", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/teams/123/image_uploads"),
					ghttp.RespondWith(http.StatusNotFound, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1ImageUploads(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a 400", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/teams/123/image_uploads"),
					ghttp.RespondWith(http.StatusBadRequest, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1ImageUploads(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when Roster successfully generates an image upload", func() {
		BeforeEach(func() {
			payloadFmt := `
				{
					"data": {
						"image_id": "%s",
						"url": "%s"
					}
				}
			`

			payload := fmt.Sprintf(payloadFmt, imageID, url)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/teams/123/image_uploads"),
					ghttp.RespondWith(http.StatusCreated, payload),
				),
			)
		})

		It("returns the image upload data", func() {
			resp, err := client.PostV1ImageUploads(context.Background(), requestParams, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Data.ImageID).To(Equal(imageID))
			Expect(resp.Data.URL).To(Equal(url))
		})
	})
})
