package roster

import (
	"context"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PostV1TeamInvitations", func() {
	var (
		server        *ghttp.Server
		client        Client
		teamID        string
		channelID     string
		requestParams v1.PostTeamInvitationsRequestParams
		err           error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()

		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})

		teamID = "123"
		channelID = "456"

		requestParams = v1.PostTeamInvitationsRequestParams{
			TeamID:    teamID,
			ChannelID: channelID,
		}

		Expect(err).NotTo((HaveOccurred()))
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/teams/123/invitations"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1TeamInvitations(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when Roster successfully creates the invitation", func() {
		BeforeEach(func() {
			payloadFmt := `
				{
					"data": {
						"channel_id": "%s",
						"team_id": "%s"
					}
				}
			`

			payload := fmt.Sprintf(payloadFmt, channelID, teamID)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/teams/123/invitations"),
					ghttp.RespondWith(http.StatusCreated, payload),
				),
			)
		})

		It("returns the created invitation", func() {
			resp, err := client.PostV1TeamInvitations(context.Background(), requestParams, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Data.ChannelID).To(Equal(channelID))
			Expect(resp.Data.TeamID).To(Equal(teamID))
		})
	})
})
