package roster

import (
	"context"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PostV1Teams", func() {
	var (
		server                                         *ghttp.Server
		client                                         Client
		userID, teamName, teamDisplayName, description string
		requestParams                                  v1.PostTeamsRequestParams
		err                                            error
	)

	BeforeEach(func() {
		server = ghttp.NewServer()

		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})

		userID = "9999"
		teamName = "the_avengers"
		teamDisplayName = "The Avengers"
		description = "# A little bit of Markdown here"

		requestParams = v1.PostTeamsRequestParams{
			UserID:              userID,
			Name:                teamName,
			DisplayName:         teamDisplayName,
			DescriptionMarkdown: &description,
		}

		Expect(err).NotTo((HaveOccurred()))
	})

	AfterEach(func() {
		server.Close()
	})

	Context("when roster returns a 500", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/users/9999/teams"),
					ghttp.RespondWith(http.StatusInternalServerError, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1Teams(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a 404", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/users/9999/teams"),
					ghttp.RespondWith(http.StatusNotFound, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1Teams(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when roster returns a 400", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/users/9999/teams"),
					ghttp.RespondWith(http.StatusBadRequest, ""),
				),
			)
		})

		It("returns an error", func() {
			resp, err := client.PostV1Teams(context.Background(), requestParams, nil)

			Expect(resp).To(BeNil())
			Expect(err).To(HaveOccurred())
		})
	})

	Context("when Roster successfully creates the featured channel", func() {
		BeforeEach(func() {
			payloadFmt := `
				{
					"data": {
						"user_id": "%s",
						"name": "%s"
					}
				}
			`

			payload := fmt.Sprintf(payloadFmt, userID, teamName)

			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPost, "/v1/users/9999/teams"),
					ghttp.RespondWith(http.StatusCreated, payload),
				),
			)
		})

		It("returns the created featured channel", func() {
			resp, err := client.PostV1Teams(context.Background(), requestParams, nil)
			Expect(err).NotTo(HaveOccurred())

			Expect(resp).NotTo(BeNil())
			Expect(resp.Data.UserID).To(Equal(userID))
			Expect(resp.Data.Name).To(Equal(teamName))
		})
	})
})
