package roster

import (
	"context"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PutV1ChannelInvitation", func() {
	var (
		server       *ghttp.Server
		client       Client
		params       v1.PutChannelInvitationRequestParams
		expectedPath string
		err          error
	)

	BeforeEach(func() {
		expectedPath = "/v1/channels/13/teams/123/invitation"
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	Context("with a bad URL", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPut, expectedPath),
					ghttp.RespondWith(http.StatusNoContent, ""),
				),
			)

			params = v1.PutChannelInvitationRequestParams{
				TeamID:    "%nothex",
				ChannelID: "%nothex",
				Accepted:  false,
			}
		})

		It("returns an error without making a request", func() {
			err = client.PutV1ChannelInvitation(context.Background(), params, nil)

			Expect(server.ReceivedRequests()).To(BeEmpty())
			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("invalid"))
		})
	})

	Context("with a valid URL", func() {
		BeforeEach(func() {
			params = v1.PutChannelInvitationRequestParams{
				TeamID:    "123",
				ChannelID: "13",
				Accepted:  true,
			}
		})

		Context("when roster returns an error", func() {
			BeforeEach(func() {
				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest(http.MethodPut, expectedPath),
						ghttp.RespondWith(http.StatusTeapot, "I'm a web-scale teapot"),
					),
				)
			})

			It("returns the error code and description", func() {
				err = client.PutV1ChannelInvitation(context.Background(), params, nil)

				Expect(server.ReceivedRequests()).To(HaveLen(1))
				Expect(err).To(HaveOccurred())
				Expect(err.Error()).To(Equal("418: I'm a web-scale teapot"))
			})
		})

		Context("when roster returns success", func() {
			BeforeEach(func() {
				accepted := true

				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest(http.MethodPut, expectedPath),
						ghttp.VerifyJSONRepresenting(v1.PutChannelInvitationRequestBody{Accepted: &accepted}),
						ghttp.RespondWith(http.StatusNoContent, ""),
					),
				)
			})

			It("returns no error", func() {
				err = client.PutV1ChannelInvitation(context.Background(), params, nil)

				Expect(server.ReceivedRequests()).To(HaveLen(1))
				Expect(err).NotTo(HaveOccurred())
			})
		})
	})
})
