package roster

import (
	"context"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/foundation/twitchclient"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/onsi/gomega/ghttp"
)

var _ = Describe("PutV1ChannelMembership", func() {
	var (
		server       *ghttp.Server
		client       Client
		params       v1.PutChannelMembershipRequestParams
		expectedPath string
		err          error
	)

	BeforeEach(func() {
		expectedPath = "/v1/channels/13/teams/123/membership"
		server = ghttp.NewServer()
		client, err = NewClient(twitchclient.ClientConf{
			Host: server.URL(),
		})
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		server.Close()
	})

	Context("with a bad URL", func() {
		BeforeEach(func() {
			server.AppendHandlers(
				ghttp.CombineHandlers(
					ghttp.VerifyRequest(http.MethodPut, expectedPath),
					ghttp.RespondWith(http.StatusNoContent, ""),
				),
			)

			params = v1.PutChannelMembershipRequestParams{
				TeamID:    "%nothex",
				ChannelID: "%nothex",
			}
		})

		It("returns an error without making a request", func() {
			err = client.PutV1ChannelMembership(context.Background(), params, nil)

			Expect(server.ReceivedRequests()).To(BeEmpty())
			Expect(err).To(HaveOccurred())
			Expect(err.Error()).To(ContainSubstring("invalid"))
		})
	})

	Context("with a valid URL", func() {
		BeforeEach(func() {
			primary := true

			params = v1.PutChannelMembershipRequestParams{
				TeamID:          "123",
				ChannelID:       "13",
				Primary:         &primary,
				RevenueRevealed: true,
				StatsRevealed:   false,
			}
		})

		Context("when roster returns an error", func() {
			BeforeEach(func() {
				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest(http.MethodPut, expectedPath),
						ghttp.RespondWith(http.StatusTeapot, "I'm a web-scale teapot"),
					),
				)
			})

			It("returns the error code and description", func() {
				err = client.PutV1ChannelMembership(context.Background(), params, nil)

				Expect(server.ReceivedRequests()).To(HaveLen(1))
				Expect(err).To(HaveOccurred())
				Expect(err.Error()).To(Equal("418: I'm a web-scale teapot"))
			})
		})

		Context("when roster returns success", func() {
			BeforeEach(func() {
				expectedReqBody := v1.PutChannelMembershipRequestBody{
					Primary:         params.Primary,
					RevenueRevealed: &params.RevenueRevealed,
					StatsRevealed:   &params.StatsRevealed,
				}

				server.AppendHandlers(
					ghttp.CombineHandlers(
						ghttp.VerifyRequest(http.MethodPut, expectedPath),
						ghttp.VerifyJSONRepresenting(expectedReqBody),
						ghttp.RespondWith(http.StatusNoContent, ""),
					),
				)
			})

			It("returns no error", func() {
				err = client.PutV1ChannelMembership(context.Background(), params, nil)

				Expect(server.ReceivedRequests()).To(HaveLen(1))
				Expect(err).NotTo(HaveOccurred())
			})
		})
	})
})
