package main

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"strings"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	"code.justin.tv/amzn/TwitchTelemetryCircuitMetrics/circuitmetrics"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"
	poller "code.justin.tv/amzn/TwitchTelemetryPollingCollector"
	"code.justin.tv/cb/roster/config"
	"code.justin.tv/cb/roster/internal/api"
	"code.justin.tv/cb/roster/internal/authorization"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/pushy"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/clients/users"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/liveline"
	"code.justin.tv/cb/roster/internal/logging"
	"code.justin.tv/cb/roster/internal/s3"
	"code.justin.tv/cb/roster/internal/s3/uploader"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/cep21/circuit"
	"github.com/cep21/circuit/closers/hystrix"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"
)

const (
	readTimeout  = 5 * time.Second
	writeTimeout = 5 * time.Second

	metricFlushInterval     = 30 * time.Second
	metricBufferSize        = 100000
	metricAggregationPeriod = 10 * time.Second
)

type simpleLogger struct{}

func (s *simpleLogger) Log(msg string, keyvals ...interface{}) {
	args := make([]interface{}, 0, len(keyvals)+1)
	args = append(args, msg)
	args = append(args, keyvals...)
	fmt.Println(args)
}

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})

	logging.SetupRollbar(os.Getenv("ROLLBAR_TOKEN"), os.Getenv("ENVIRONMENT"))
	config.Load()
}

func main() {
	env := config.Environment

	decoder, err := authorization.NewDecoder(os.Getenv("ECC_PUBLIC_KEY_PATH"))
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate authorization decoder")
		return
	}

	redis, err := cache.NewRedis(os.Getenv("REDIS_HOST"), os.Getenv("REDIS_PORT"))
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate redis cache client")
		return
	}

	dbMaster, err := db.NewClient(db.ClientConfig{
		Host:         config.Values.DB.Master.Host,
		Port:         config.Values.DB.Master.Port,
		DBName:       config.Values.DB.Master.Name,
		SSLMode:      config.Values.DB.Master.SSLMode,
		User:         config.Values.DB.Master.User,
		Password:     config.Secrets.DB.Password,
		MaxConns:     config.Values.DB.Master.MaxConns,
		MaxIdleConns: config.Values.DB.Master.MaxIdleConns,
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new master db client")
		return
	}

	dbReplica, err := db.NewClient(db.ClientConfig{
		Host:         config.Values.DB.Replica.Host,
		Port:         config.Values.DB.Replica.Port,
		DBName:       config.Values.DB.Replica.Name,
		SSLMode:      config.Values.DB.Replica.SSLMode,
		User:         config.Values.DB.Replica.User,
		Password:     config.Secrets.DB.Password,
		MaxConns:     config.Values.DB.Replica.MaxConns,
		MaxIdleConns: config.Values.DB.Master.MaxIdleConns,
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new replica db client")
		return
	}

	// Telemetry set up
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}

	processID := &identifier.ProcessIdentifier{
		Service: "Roster",
		Stage:   env,
		Region:  config.Values.AWSRegion,
		Machine: hostname,
	}

	sender := cw.NewUnbuffered(processID, nil)
	baseObserver := telemetry.NewBufferedAggregator(metricFlushInterval, metricBufferSize, metricAggregationPeriod, sender, &simpleLogger{})

	sampleReporter := telemetry.SampleReporter{
		SampleBuilder:  telemetry.SampleBuilder{ProcessIdentifier: *processID},
		SampleObserver: baseObserver,
	}

	// Circuit set up
	circuitManager := circuit.Manager{}
	metrics := circuitmetrics.CommandFactory{
		Reporter: sampleReporter,
	}

	hystrixConfig := hystrix.Factory{
		CreateConfigureCloser: []func(circuitName string) hystrix.ConfigureCloser{
			func(circuitName string) hystrix.ConfigureCloser {
				attempts := int64(1)
				if strings.HasPrefix(circuitName, "users") {
					attempts = 10
				}
				return hystrix.ConfigureCloser{
					HalfOpenAttempts:             attempts,
					RequiredConcurrentSuccessful: attempts,
				}
			},
		},
		CreateConfigureOpener: []func(circuitName string) hystrix.ConfigureOpener{
			func(circuitName string) hystrix.ConfigureOpener {
				// defaults are fine for this
				return hystrix.ConfigureOpener{}
			},
		},
	}

	circuitManager.DefaultCircuitProperties = append(circuitManager.DefaultCircuitProperties,
		metrics.CommandProperties,
		hystrixConfig.Configure,
	)

	collector := metrics.ConcurrencyCollector(&circuitManager)
	go collector.Start()

	usersClient, err := users.NewClient(os.Getenv("USERS_SERVICE_HOST"), createUserCircuits(&circuitManager))
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new users service client")
		return
	}

	livelineClient := liveline.NewClient()

	awsSession, err := newAWSSession()
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new aws session")
		return
	}

	uploadsBucket := os.Getenv("S3_UPLOADS_BUCKET")

	// NOTE: receiver currently cannout use S2S2 as designed because of https://jira.twitch.com/browse/SECDEV-2656
	// uncomment this section when SECDEV-2656 is resolved
	//
	// s2s2Client, err := s2s2.New(&s2s2.Options{
	// 	Config: &c7s.Config{
	// 		ClientServiceName: config.Values.S2S2ServiceName,
	// 	},
	// })
	// if err != nil {
	// 	log.WithError(err).Fatal("failed to instantiate S2S2 client")
	// 	return
	// }

	// s2sHTTPClient := &http.Client{
	// 	Transport: s2s2Client.RoundTripper(&http.Client{}),
	// }

	s2sHTTPClient := &http.Client{}

	server := &http.Server{
		Addr: ":8000",
		Handler: api.NewServer(&api.ServerParams{
			AuthDecoder:      decoder,
			Cache:            redis,
			DBWriter:         dbMaster,
			DBReader:         dbReplica,
			S3:               s3.NewClient(awsSession, uploadsBucket, os.Getenv("S3_ATTACHMENTS_BUCKET")),
			S3Uploader:       uploader.NewClient(awsSession, uploadsBucket),
			Users:            usersClient,
			Pushy:            pushy.NewDartClient(s2sHTTPClient, sampleReporter),
			Liveline:         livelineClient,
			TelemetryHandler: &telemetryhook.Client{Reporter: sampleReporter},
		}),
		ReadTimeout:  readTimeout,
		WriteTimeout: writeTimeout,
	}

	goStatsPoller := poller.NewGoStatsPollingCollector(10*time.Second, &sampleReporter.SampleBuilder, sampleReporter.SampleObserver, &simpleLogger{})
	goStatsPoller.Start()

	go func() {
		log.Info("server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("server failed fatally while listening")
		}
	}()

	httputil.Graceful(context.Background(), server, goStatsPoller, collector)
}

func newAWSSession() (*session.Session, error) {
	config := &aws.Config{
		Region: aws.String(os.Getenv("AWS_REGION")),
	}

	if os.Getenv("ENVIRONMENT") == "development" {
		config.Credentials = credentials.NewSharedCredentials("", os.Getenv("AWS_PROFILE"))
	}

	return session.NewSession(config)
}

func createUserCircuits(cm *circuit.Manager) users.UserCircuits {
	return users.UserCircuits{
		Get: cm.MustCreateCircuit("users.get", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout:               time.Millisecond * 80,
				MaxConcurrentRequests: 500,
			},
		}),
		GetAll: cm.MustCreateCircuit("users.getAll", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout:               time.Millisecond * 80,
				MaxConcurrentRequests: 100,
			},
		}),
		GetByIDAndParams: cm.MustCreateCircuit("users.getByIDAndParams", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout:               time.Millisecond * 80,
				MaxConcurrentRequests: 100,
			},
		}),
		Set: cm.MustCreateCircuit("users.set", circuit.Config{
			Execution: circuit.ExecutionConfig{
				Timeout:               time.Millisecond * 500,
				MaxConcurrentRequests: 100,
			},
		}),
	}
}
