package main

import (
	"context"
	"fmt"
	"net/http"
	"os"
	"time"

	identifier "code.justin.tv/amzn/TwitchProcessIdentifier"
	telemetry "code.justin.tv/amzn/TwitchTelemetry"
	cw "code.justin.tv/amzn/TwitchTelemetryCloudWatchMetricsSender"
	poller "code.justin.tv/amzn/TwitchTelemetryPollingCollector"
	_ "github.com/lib/pq"
	log "github.com/sirupsen/logrus"

	"code.justin.tv/cb/roster/config"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/pdms"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/eventbus"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/logging"
	"code.justin.tv/cb/roster/internal/worker"
)

const (
	readTimeout  = 10 * time.Second
	writeTimeout = 10 * time.Second

	metricFlushInterval     = 30 * time.Second
	metricBufferSize        = 100000
	metricAggregationPeriod = 10 * time.Second
)

type simpleLogger struct{}

func (s *simpleLogger) Log(msg string, keyvals ...interface{}) {
	args := make([]interface{}, 0, len(keyvals)+1)
	args = append(args, msg)
	args = append(args, keyvals...)
	fmt.Println(args)
}

func init() {
	log.SetFormatter(&log.TextFormatter{
		FullTimestamp: true,
	})

	logging.SetupRollbar(os.Getenv("ROLLBAR_TOKEN"), os.Getenv("ENVIRONMENT"))
	config.Load()
}

func main() {
	redis, err := cache.NewRedis(os.Getenv("REDIS_HOST"), os.Getenv("REDIS_PORT"))
	if err != nil {
		log.WithError(err).Fatal("failed to instantiate redis cache client")
		return
	}

	dbMaster, err := db.NewClient(db.ClientConfig{
		Host:         config.Values.DB.Master.Host,
		Port:         config.Values.DB.Master.Port,
		DBName:       config.Values.DB.Master.Name,
		SSLMode:      config.Values.DB.Master.SSLMode,
		User:         config.Values.DB.Master.User,
		Password:     config.Secrets.DB.Password,
		MaxConns:     config.Values.DB.Master.MaxConns,
		MaxIdleConns: config.Values.DB.Master.MaxIdleConns,
	})

	if err != nil {
		log.WithError(err).Fatal("failed to instantiate new master db client")
		return
	}

	// Telemetry set up
	env := config.Environment
	hostname, err := os.Hostname()
	if err != nil {
		hostname = "unknown"
	}

	processID := &identifier.ProcessIdentifier{
		Service: "Roster",
		Stage:   env,
		Region:  config.Values.AWSRegion,
		Machine: hostname,
	}

	sender := cw.NewUnbuffered(processID, nil)
	baseObserver := telemetry.NewBufferedAggregator(metricFlushInterval, metricBufferSize, metricAggregationPeriod, sender, &simpleLogger{})

	sampleReporter := telemetry.SampleReporter{
		SampleBuilder:  telemetry.SampleBuilder{ProcessIdentifier: *processID},
		SampleObserver: baseObserver,
	}

	pdmsClient := pdms.NewClient(pdms.ClientConfig{
		CallerRoleArn: os.Getenv("PDMS_CALLER_ROLE_ARN"),
		LambdaArn:     os.Getenv("PDMS_LAMBDA_ARN"),
	})

	eventbusErr := eventbus.Initialise(&eventbus.HandlerParams{
		Cache:      redis,
		DBWriter:   dbMaster,
		PDMSClient: pdmsClient,
		Reporter:   sampleReporter,
	})

	if eventbusErr != nil {
		log.WithError(eventbusErr).Fatal("failed to instantiate eventbus client")
		return
	}

	server := &http.Server{
		Addr: ":8000",
		Handler: worker.NewServer(&worker.ServerParams{
			Cache:            redis,
			DBWriter:         dbMaster,
			TelemetryHandler: &telemetryhook.Client{Reporter: sampleReporter},
		}),
		ReadTimeout:  readTimeout,
		WriteTimeout: writeTimeout,
	}

	log.Info("server listening on http://localhost", server.Addr)
	goStatsPoller := poller.NewGoStatsPollingCollector(10*time.Second, &sampleReporter.SampleBuilder, sampleReporter.SampleObserver, &simpleLogger{})
	goStatsPoller.Start()

	go func() {
		log.Info("server listening on http://localhost", server.Addr)

		if err := server.ListenAndServe(); err != http.ErrServerClosed {
			log.WithError(err).Fatal("server failed fatally while listening")
		}
	}()

	httputil.Graceful(context.Background(), server, goStatsPoller, nil)
}
