package config

import (
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"runtime"

	log "github.com/sirupsen/logrus"
	"gopkg.in/yaml.v2"
)

// These constants are environment types that could be used for comparison
// with the configured environment, outside of this package.
const (
	Development = "development"
	Staging     = "staging"
	Production  = "production"
)

const (
	devConfigAndSecretsPath = "development.yaml"
	configPath              = "/var/app/current/config.yaml"
	// The application secrets YAML path is defined in the Elastic Beanstalk
	// extensions for the Sandstorm-Agent:
	//
	// .ebextensions/data/sandstorm-agent/conf.d/03-secrets.conf
	// .ebextensions/data/sandstorm-agent/templates.d/secrets.yaml
	secretsPath = "/var/app/secrets.yaml"
)

var (
	// Environment is a global variable of the application environment
	// according to the environment variable `ENVIRONMENT`.
	Environment string

	// Values is a global variable containing the fields of all
	// environment-specific configuration values for the application.
	Values values

	// Secrets is a global variable containing the fields of all
	// environment-specific secrets for the application.
	Secrets secrets
)

// values contains the non-secret configuration fields for the application.
//
// After defining a field within the values structure,
// the attribute MUST be defined in the following YAML files
// for decoding during application start-up:
//
// * config/development.yaml
// * config/staging.yaml
// * config/production.yaml
//
// Example:
//
// type values struct {
// 	DB struct {
// 		Addresses struct {
// 			Master  string
// 			Replica string
// 		}
// 	}
// }
//
// development.yaml:
//   db:
//     addresses:
//       master: http://0.0.0.0
//       replica: http://1.2.3.4
type values struct {
	SNSTopics struct {
		Pushy string `yaml:"pushy_sns_arn"`
	} `yaml:"sns"`

	ServiceHosts struct {
		Dart string `yaml:"dart"`
	} `yaml:"service_hosts"`

	DB struct {
		Master struct {
			Host         string `yaml:"host"`
			Port         string `yaml:"port"`
			Name         string `yaml:"name"`
			SSLMode      string `yaml:"sslmode"`
			User         string `yaml:"user"`
			MaxConns     int    `yaml:"max_conns"`
			MaxIdleConns int    `yaml:"max_idle_conns"`
		} `yaml:"master"`
		Replica struct {
			Host         string `yaml:"host"`
			Port         string `yaml:"port"`
			Name         string `yaml:"name"`
			SSLMode      string `yaml:"sslmode"`
			User         string `yaml:"user"`
			MaxConns     int    `yaml:"max_conns"`
			MaxIdleConns int    `yaml:"max_idle_conns"`
		} `yaml:"replica"`
	} `yaml:"db"`

	AWSRegion       string `yaml:"aws_region"`
	S2S2ServiceName string `yaml:"s2s2_service_name"`
}

// secrets contains the secret configuration values for the application.
//
// For development ONLY, after defining a field in the secrets structure,
// the attribute MUST be defined in config/development.yaml.
//
// For staging or production, a secret's attribute key and template value
// MUST be added to the following file:
//
// .ebextensions/data/sandstorm-agent/templates.d/secrets
//
// for the Sandstorm-Agent to insert first and then for unmarshaling
// during the start-up of the application.
// The secret MUST also be created within Sandstorm under the same key.
//
// Example:
//
// type secrets struct {
// 	DB struct{
// 		Password string
// 	}
// }
//
// development.yaml:
//   db:
//     password: hunter2
//
// .ebextensions/data/sandstorm-agent/templates.d/secrets:
//   db:
//     password: {{ key "creator-business/roster/{ENVIRONMENT}/password" }}
type secrets struct {
	DB struct {
		Password string
	} `yaml:"db"`
}

func init() {
	Environment = os.Getenv("ENVIRONMENT")
	if Environment != Production && Environment != Staging {
		Environment = Development
	}
}

// Load decodes both non-secret and secret configuration values from YAML.
//
// In development, the `config/development.yaml` file is loaded to
// populate BOTH the Values and Secrets global variables.
//
// In staging or production, the corresponding non-secret config YAML
// (config/staging.yaml or config/production.yaml) is added
// to the /var/app/current directory as config.yaml during
// the deployment process (See: scripts/deploy_beanstalk.sh).
//
// The secret config YAML is created by the Sandstorm-Agent
// at /var/app/secrets.yaml
// (See: .ebextensions/data/sandstorm-agent/conf.d/03-secrets.conf)
func Load() {
	configFilePath := configPath
	secretsFilePath := secretsPath

	if Environment == Development {
		_, filename, _, ok := runtime.Caller(0)
		if !ok {
			log.Panic("No caller information to get config file name")
		}

		filePath := fmt.Sprint(path.Dir(filename), "/", devConfigAndSecretsPath)
		configFilePath = filePath
		secretsFilePath = filePath
	}

	Values = loadValues(configFilePath)
	Secrets = loadSecrets(secretsFilePath)
}

func loadValues(filePath string) values {
	v := values{}

	log.Info("Loading config from ", filePath)

	bytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		log.WithError(err).Panic("Failed to open config YAML")
	}

	err = yaml.Unmarshal(bytes, &v)
	if err != nil {
		log.WithError(err).Panic("Failed to decode config YAML")
	}

	return v
}

func loadSecrets(filePath string) secrets {
	s := secrets{}

	log.Info("Loading secrets from ", filePath)

	bytes, err := ioutil.ReadFile(filePath)
	if err != nil {
		log.WithError(err).Panic("Failed to open configuration secrets YAML")
		return s
	}

	err = yaml.Unmarshal(bytes, &s)
	if err != nil {
		log.WithError(err).Panic("Failed to decode configuration secrets YAML")
	}

	return s
}
