package api

import (
	"context"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/db"
	log "github.com/sirupsen/logrus"
)

func (s *Server) fetchCachedTeam(ctx context.Context, teamID string) []byte {
	json, err := s.cache.GetTeam(ctx, teamID)
	if err != nil {
		switch err {
		case cache.ErrNoTeam:
		default:
			log.WithError(err).WithField("team_id", teamID).Warn("failed to fetch team from cache")
		}
		return nil
	}

	return []byte(json)
}

func (s *Server) cacheTeam(ctx context.Context, teamID string, resp v1.GetTeamResponse) {
	err := s.cache.SetTeam(ctx, teamID, jsonString(resp))
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"team_id":  teamID,
			"response": resp,
		}).Warn("failed to cache team")
	}
}

func (s *Server) expireCachedTeam(ctx context.Context, teamID string) {
	err := s.cache.ClearTeam(ctx, teamID)
	if err != nil {
		log.WithError(err).WithField("team_id", teamID).Warn("failed to clear cached team")
	}
}

func (s *Server) fetchCachedTeams(ctx context.Context, query cache.TeamsQuery) []byte {
	json, err := s.cache.GetTeams(ctx, query)
	if err != nil {
		switch err {
		case cache.ErrNoTeams:
		default:
			log.WithError(err).WithField("query", query).Warn("failed to fetch teams from cache")
		}
		return nil
	}

	return []byte(json)
}

func (s *Server) cacheTeams(ctx context.Context, query cache.TeamsQuery, resp v1.GetTeamsResponse) {
	err := s.cache.SetTeams(ctx, query, jsonString(resp))
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"query":    query,
			"response": resp,
		}).Warn("failed to cache teams")
	}
}

func (s *Server) expireCachedTeams(ctx context.Context) {
	err := s.cache.ClearAllTeams(ctx)
	if err != nil {
		log.WithError(err).Warn("failed to clear cached teams")
	}
}

func (s *Server) fetchCachedChannelMemberships(ctx context.Context, channelID string) []byte {
	json, err := s.cache.GetChannelMemberships(ctx, channelID)
	if err != nil {
		switch err {
		case cache.ErrNoChannelMemberships:
		default:
			log.WithError(err).WithField("channel_id", channelID).Warn("failed to fetch channel memberships from cache")
		}
		return nil
	}

	return []byte(json)
}

func (s *Server) cacheChannelMemberships(ctx context.Context, channelID string, resp v1.GetChannelMembershipsResponse) {
	err := s.cache.SetChannelMemberships(ctx, channelID, jsonString(resp))
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"channel_id": channelID,
			"response":   resp,
		}).Warn("failed to cache channel memberships")
	}
}

func (s *Server) expireCachedChannelMemberships(ctx context.Context, channelID string) {
	err := s.cache.ClearChannelMemberships(ctx, channelID)
	if err != nil {
		log.WithError(err).WithField("channel_id", channelID).Warn("failed to clear cached channel memberships")
	}
}

func (s *Server) expireAllCachedChannelMemberships(ctx context.Context, memberships []db.Membership) {
	for _, membership := range memberships {
		go s.expireCachedChannelMemberships(ctx, membership.ChannelID)
	}
}

func (s *Server) expireCachedChannelMembershipsByTeam(ctx context.Context, teamID string) {
	memberships, err := s.dbReader.GetTeamMemberships(ctx, teamID, nil)
	if err != nil {
		log.WithError(err).WithField("team_id", teamID).Error("failed to get team memberships from db to clear cache")
		return
	}

	s.expireAllCachedChannelMemberships(ctx, memberships)
}

func (s *Server) fetchCachedTeamMemberships(ctx context.Context, query cache.TeamMembershipsQuery) []byte {
	json, err := s.cache.GetTeamMemberships(ctx, query)
	if err != nil {
		switch err {
		case cache.ErrNoTeamMemberships:
		default:
			log.WithError(err).WithField("team_id", query.ID).Warn("failed to fetch team memberships from cache")
		}
		return nil
	}

	return []byte(json)
}

func (s *Server) cacheTeamMemberships(ctx context.Context, query cache.TeamMembershipsQuery, resp v1.GetTeamMembershipsResponse) {
	err := s.cache.SetTeamMemberships(ctx, query, jsonString(resp))
	if err != nil {
		log.WithError(err).WithFields(log.Fields{
			"team_id":  query.ID,
			"response": resp,
		}).Warn("failed to cache team memberships")
	}
}

func (s *Server) expireCachedTeamMemberships(ctx context.Context, teamID string) {
	err := s.cache.ClearAllTeamMembershipsForTeam(ctx, teamID)
	if err != nil {
		log.WithError(err).WithField("team_id", teamID).Warn("failed to clear cached team memberships")
	}
}
