package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("DeleteV1FeaturedChannel", func() {
	var (
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		server   *Server
		recorder *httptest.ResponseRecorder
		users    *mocks.Users

		teamID, channelID string
		req               *http.Request
		err               error
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamID = "123"
		channelID = "456"

		path := fmt.Sprintf("/v1/teams/%s/featured_channels/%s", teamID, channelID)
		req, err = http.NewRequest(http.MethodDelete, path, nil)
		Expect(err).NotTo(HaveOccurred())
	})

	It("fails with Not Found when the team does not exist", func() {
		dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)

		server.ServeHTTP(recorder, req)

		dbReader.AssertExpectations(GinkgoT())
		Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
	})

	It("fails with Internal Server Error when the database errors while querying for the team", func() {
		dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("Database error"))

		server.ServeHTTP(recorder, req)

		dbReader.AssertExpectations(GinkgoT())
		Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
	})

	Context("when the team is found", func() {
		var foundTeam db.Team

		BeforeEach(func() {
			foundTeam = db.Team{
				ID:     teamID,
				Name:   "some-team-name",
				UserID: "team-owner-id",
			}
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(foundTeam, nil)
		})

		It("fails with Not Found when the channel is not found in Users Service", func() {
			users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(nil, &channels.ErrChannelNotFound{})

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
		})

		It("fails with Internal Server Error when request to the Users Service fails", func() {
			users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&channels.Channel{}, errors.New("some error"))

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
		})

		Context("when the Users Service return the Channel", func() {
			BeforeEach(func() {
				foundChannel := channels.Channel{
					ID: 456,
				}
				users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&foundChannel, nil)
			})

			It("fails with Internal Server Error when the database errors while deleting the featured channel", func() {
				dbWriter.On(
					"DeleteFeaturedChannel",
					mock.Anything,
					teamID,
					channelID,
				).Return(errors.New("Internal server error"))

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				users.AssertExpectations(GinkgoT())
				dbWriter.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			})

			It("fails with Not Found when the database returns a not found error", func() {
				dbWriter.On("DeleteFeaturedChannel", mock.Anything, teamID, channelID).Return(db.ErrNoRowFoundForDeletion)

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				users.AssertExpectations(GinkgoT())
				dbWriter.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
			})

			It("succeeds with No Content when the featured channel is successfully deleted", func() {
				dbWriter.On("DeleteFeaturedChannel", mock.Anything, teamID, channelID).Return(nil)

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				users.AssertExpectations(GinkgoT())
				dbWriter.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
			})
		})
	})
})
