package api

import (
	"context"
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

// DELETE /v1/teams/:team_id/channels/:channel_id/membership or
// DELETE /v1/channels/:channel_id/teams/:team_id/membership
func (s *Server) deleteV1Membership(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()

	teamID := ctx.Value(contextKeyTeamID).(string)
	channelID := ctx.Value(contextKeyChannelID).(string)

	_, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to query team", err)
		}
		return
	}

	channel, err := s.users.Get(ctx, channelID, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.NotFound(err.Error())
		default:
			jsonWriter.InternalServerError("users_service: failed to look up channel", err)
		}
		return
	}

	if strconv.Itoa(channel.PrimaryTeamID) == teamID {
		err = s.users.Set(ctx, models.UpdateChannelProperties{
			ID:                  uint64(channel.ID),
			PrimaryTeamID:       nil,
			DeletePrimaryTeamID: true,
		}, nil)

		if err != nil {
			jsonWriter.InternalServerError("users_service: failed to delete channel primary team ID", err)
			return
		}
	}

	err = s.dbWriter.DeleteMembership(ctx, teamID, channelID)
	if err != nil {
		switch err {
		case db.ErrNoRowFoundForDeletion:
			jsonWriter.NotFound(fmt.Sprintf("membership of channel with id %s and team with id %s not found", channelID, teamID))
		default:
			jsonWriter.InternalServerError("db: failed to delete membership", err)
		}
		return
	}

	go s.expireCachedChannelMemberships(context.Background(), channelID)
	go s.expireCachedTeamMemberships(context.Background(), teamID)

	w.WriteHeader(http.StatusNoContent)
}
