package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("DeleteV1Membership", func() {
	var (
		mockedCache *mocks.Cache
		dbWriter    *mocks.DBWriter
		dbReader    *mocks.DBReader
		server      *Server
		recorder    *httptest.ResponseRecorder
		users       *mocks.Users

		teamID, channelID string
		req               *http.Request
		err               error
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		mockedCache = &mocks.Cache{}
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			Cache:            mockedCache,
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamID = "123"
		channelID = "456"
	})

	Describe("Deleting membership from team's perspective", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", teamID, channelID)
			req, err = http.NewRequest(http.MethodDelete, path, nil)
			Expect(err).NotTo(HaveOccurred())
		})

		It("fails with Not Found when the team does not exist", func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
		})

		It("fails with Internal Server Error when the database errors while querying for the team", func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("Database error"))

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
		})

		Context("when the team is found", func() {
			var foundTeam db.Team

			BeforeEach(func() {
				foundTeam = db.Team{
					ID:     teamID,
					Name:   "some-team-name",
					UserID: "team-owner-id",
				}
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(foundTeam, nil)
			})

			It("fails with Not Found when the channel is not found in Users Service", func() {
				users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(nil, &channels.ErrChannelNotFound{})

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				users.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
			})

			It("fails with Internal Server Error when request to the Users Service fails", func() {
				users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&channels.Channel{}, errors.New("some error"))

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				users.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			})

			Context("when the Users Service return the Channel", func() {
				Context("when the primary team id matches the given team id", func() {
					var expectedUpdateChannelProperties models.UpdateChannelProperties

					BeforeEach(func() {
						foundChannel := channels.Channel{
							ID:            456,
							PrimaryTeamID: 123,
						}
						users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&foundChannel, nil)
						expectedUpdateChannelProperties = models.UpdateChannelProperties{
							ID:                  456,
							PrimaryTeamID:       nil,
							DeletePrimaryTeamID: true,
						}
					})

					It("fails with Internal Server Error when patching the Channel fails", func() {
						users.On("Set", mock.Anything, expectedUpdateChannelProperties, (*twitchclient.ReqOpts)(nil)).Return(errors.New("som error"))

						server.ServeHTTP(recorder, req)

						dbReader.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					})

					Context("when patching the Channel succeeds", func() {
						BeforeEach(func() {
							users.On("Set", mock.Anything, expectedUpdateChannelProperties, (*twitchclient.ReqOpts)(nil)).Return(nil)
						})

						It("fails with Internal Server Error when the database errors while deleting the membership", func() {
							dbWriter.On(
								"DeleteMembership",
								mock.Anything,
								teamID,
								channelID,
							).Return(errors.New("Internal server error"))

							server.ServeHTTP(recorder, req)

							dbReader.AssertExpectations(GinkgoT())
							users.AssertExpectations(GinkgoT())
							dbWriter.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
						})

						It("fails with Not Found when the database returns a not found error", func() {
							dbWriter.On("DeleteMembership", mock.Anything, teamID, channelID).Return(db.ErrNoRowFoundForDeletion)

							server.ServeHTTP(recorder, req)

							dbReader.AssertExpectations(GinkgoT())
							users.AssertExpectations(GinkgoT())
							dbWriter.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
						})

						Context("When the membership is successfully deleted", func() {
							BeforeEach(func() {
								mockedCache.On("ClearChannelMemberships", mock.Anything, channelID).Return(nil)
								mockedCache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
							})

							It("succeeds with No Content", func() {
								dbWriter.On("DeleteMembership", mock.Anything, teamID, channelID).Return(nil)

								server.ServeHTTP(recorder, req)

								dbReader.AssertExpectations(GinkgoT())
								users.AssertExpectations(GinkgoT())
								dbWriter.AssertExpectations(GinkgoT())
								Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
							})
						})
					})
				})

				Context("when the primary team id does not match the given team id", func() {
					BeforeEach(func() {
						foundChannel := channels.Channel{
							ID:            456,
							PrimaryTeamID: 789,
						}
						users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&foundChannel, nil)
					})

					It("fails with Internal Server Error when the database errors while deleting the membership", func() {
						dbWriter.On("DeleteMembership", mock.Anything, teamID, channelID).
							Return(errors.New("Internal server error"))

						server.ServeHTTP(recorder, req)

						dbReader.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())
						dbWriter.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					})

					It("fails with Not Found when the database returns a not found error", func() {
						dbWriter.On("DeleteMembership", mock.Anything, teamID, channelID).Return(db.ErrNoRowFoundForDeletion)

						server.ServeHTTP(recorder, req)

						dbReader.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())
						dbWriter.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
					})

					Context("When the membership is successfully deleted", func() {
						BeforeEach(func() {
							mockedCache.On("ClearChannelMemberships", mock.Anything, channelID).Return(nil)
							mockedCache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
						})

						It("succeeds with No Content", func() {
							dbWriter.On("DeleteMembership", mock.Anything, teamID, channelID).Return(nil)

							server.ServeHTTP(recorder, req)

							dbReader.AssertExpectations(GinkgoT())
							users.AssertExpectations(GinkgoT())
							dbWriter.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
						})
					})
				})
			})
		})
	})

	Describe("Deleting membership from channel's perspective", func() {
		var foundTeam db.Team
		var expectedUpdateChannelProperties models.UpdateChannelProperties

		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/teams/%s/membership", channelID, teamID)
			req, err = http.NewRequest(http.MethodDelete, path, nil)
			Expect(err).NotTo(HaveOccurred())
			foundTeam = db.Team{
				ID:     teamID,
				Name:   "some-team-name",
				UserID: "team-owner-id",
			}
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(foundTeam, nil)
			foundChannel := channels.Channel{
				ID:            456,
				PrimaryTeamID: 123,
			}
			users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&foundChannel, nil)
			expectedUpdateChannelProperties = models.UpdateChannelProperties{
				ID:                  456,
				PrimaryTeamID:       nil,
				DeletePrimaryTeamID: true,
			}
			users.On("Set", mock.Anything, expectedUpdateChannelProperties, (*twitchclient.ReqOpts)(nil)).Return(nil)

			mockedCache.On("ClearChannelMemberships", mock.Anything, channelID).Return(nil)
			mockedCache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
			dbWriter.On("DeleteMembership", mock.Anything, teamID, channelID).Return(nil)
		})

		It("calls the same handler and succeeds with No Content when the membership is successfully deleted", func() {
			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			users.AssertExpectations(GinkgoT())
			dbWriter.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
		})
	})
})
