package api

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/models"
	log "github.com/sirupsen/logrus"
)

func (s *Server) deleteV1Team(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	_, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError(fmt.Sprintf("db: failed to query team %s", teamID), err)
		}
		return
	}

	memberships, err := s.dbReader.GetTeamMemberships(ctx, teamID, nil)
	if err != nil {
		jsonWriter.InternalServerError(fmt.Sprintf("db: failed to query memberships for team %s", teamID), err)
		return
	}

	// There is a race condition in which one or more channels sets this team to be primary,
	// BEFORE this team is deleted.
	//
	// We accept the risk of this team-channel state mismatch, considering the rarity of team deletions.
	channelIDs, err := s.filterChannelIDsForPrimaryTeamMembership(ctx, teamID, memberships)
	if err != nil {
		msg := fmt.Sprintf("users service: failed to query channels for team %s memberships", teamID)
		jsonWriter.InternalServerError(msg, err)
		return
	}

	err = s.deletePrimaryTeamIDForChannelIDs(ctx, teamID, channelIDs)
	if err != nil {
		msg := fmt.Sprintf("users service: failed to delete primary team ID for one or more channels of team %s", teamID)
		jsonWriter.InternalServerError(msg, err)
		return
	}

	err = s.dbWriter.DeleteTeam(ctx, teamID)
	if err != nil {
		msg := fmt.Sprintf("db: failed to delete team %s and associated relations", teamID)
		jsonWriter.InternalServerError(msg, err)
		return
	}

	go s.expireCachedTeam(context.Background(), teamID)
	go s.expireCachedTeams(context.Background())
	go s.expireAllCachedChannelMemberships(context.Background(), memberships)
	go s.expireCachedTeamMemberships(context.Background(), teamID)

	w.WriteHeader(http.StatusNoContent)
}

func (s *Server) filterChannelIDsForPrimaryTeamMembership(ctx context.Context, teamID string, memberships []db.Membership) ([]int, error) {
	channelIDs := make([]string, len(memberships))
	for i, membership := range memberships {
		channelIDs[i] = membership.ChannelID
	}

	result, err := s.users.GetAll(ctx, channelIDs, nil)
	if err != nil {
		return nil, err
	}

	primaryTeamMemberIDs := []int{}
	for _, channel := range result.Results {
		if strconv.Itoa(channel.PrimaryTeamID) == teamID {
			primaryTeamMemberIDs = append(primaryTeamMemberIDs, channel.ID)
		}
	}

	return primaryTeamMemberIDs, nil
}

func (s *Server) deletePrimaryTeamIDForChannelIDs(ctx context.Context, teamID string, channelIDs []int) error {
	var wg sync.WaitGroup
	failureChan := make(chan int, len(channelIDs))
	successChan := make(chan int, len(channelIDs))

	for _, channelID := range channelIDs {
		wg.Add(1)

		go func(id int) {
			defer wg.Done()

			err := s.users.Set(ctx, models.UpdateChannelProperties{
				ID:                  uint64(id),
				PrimaryTeamID:       nil,
				DeletePrimaryTeamID: true,
			}, nil)
			if err != nil {
				failureChan <- id
				return
			}

			successChan <- id
		}(channelID)
	}

	wg.Wait()

	close(failureChan)
	close(successChan)

	failures := []int{}
	for failedID := range failureChan {
		failures = append(failures, failedID)
	}

	successes := []int{}
	for succeededID := range successChan {
		successes = append(successes, succeededID)
	}

	if len(failures) > 0 {
		msg := fmt.Sprintf("users service: failed to delete primary team ID for one or more channels (team %s)", teamID)

		log.WithFields(log.Fields{
			"team_id":                              teamID,
			"all_channel_ids":                      channelIDs,
			"failed_to_delete_for_channel_ids":     failures,
			"successfully_deleted_for_channel_ids": successes,
		}).Error(msg + " -- action required to reinstate primary team ID!")

		return errors.New(msg)
	}

	return nil
}
