package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("DeleteV1TeamInvitation", func() {
	var (
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		server   *Server
		recorder *httptest.ResponseRecorder

		teamID, channelID string
		req               *http.Request
		err               error
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamID = "123"
		channelID = "456"

		path := fmt.Sprintf("/v1/teams/%s/channels/%s/invitation", teamID, channelID)
		req, err = http.NewRequest(http.MethodDelete, path, nil)
		Expect(err).NotTo(HaveOccurred())
	})

	It("fails with Not Found when the team does not exist", func() {
		dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)

		server.ServeHTTP(recorder, req)

		dbReader.AssertExpectations(GinkgoT())
		Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
	})

	It("fails with Internal Server Error when the database errors while querying for the team", func() {
		dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("Database error"))

		server.ServeHTTP(recorder, req)

		dbReader.AssertExpectations(GinkgoT())
		Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
	})

	Context("when the team is found", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
				ID:     teamID,
				Name:   "some-team-name",
				UserID: "team-owner-id",
			}, nil)
		})

		It("fails with Internal Server Error when the database errors while deleting the invitation", func() {
			dbWriter.On("DeleteInvitation", mock.Anything, teamID, channelID).Return(errors.New("error"))

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			dbWriter.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
		})

		It("fails with Not Found when the database returns a not found error", func() {
			dbWriter.On("DeleteInvitation", mock.Anything, teamID, channelID).Return(db.ErrNoRowFoundForDeletion)

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			dbWriter.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
		})

		It("succeeds with No Content when the invitation is successfully deleted", func() {
			dbWriter.On("DeleteInvitation", mock.Anything, teamID, channelID).Return(nil)

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			dbWriter.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
		})
	})
})
