package api

import (
	"errors"
	"net/http"
	"net/http/httptest"
	"path"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("DeleteV1Team", func() {
	var (
		cache    *mocks.Cache
		dbReader *mocks.DBReader
		dbWriter *mocks.DBWriter
		users    *mocks.Users
		server   *Server
		recorder *httptest.ResponseRecorder
		teamID   string
		req      *http.Request
		err      error
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		cache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}
		dbWriter = &mocks.DBWriter{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			Cache:            cache,
			DBReader:         dbReader,
			DBWriter:         dbWriter,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamID = "9000"

		req, err = http.NewRequest(http.MethodDelete, path.Join("/v1/teams", teamID), nil)
		Expect(err).NotTo(HaveOccurred())
	})

	AfterEach(func() {
		dbReader.AssertExpectations(GinkgoT())
		dbWriter.AssertExpectations(GinkgoT())
		users.AssertExpectations(GinkgoT())
	})

	Context("when the team does not exist", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
		})

		It("returns Not Found when the team does not exist", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
		})
	})

	Context("when DB fails to query for the team", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("💩"))
		})

		It("returns Not Found when the team does not exist", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			Expect(recorder.Body.String()).To(ContainSubstring("failed to query team"))
		})
	})

	Context("when the team is found", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
				ID: teamID,
			}, nil)
		})

		Context("when DB fails to query for the team's memberships", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).Return(nil, errors.New("🔥"))
			})

			It("returns Internal Server Error", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
				Expect(recorder.Body.String()).To(ContainSubstring("failed to query memberships for team"))
			})
		})

		Context("when DB returns memberships", func() {
			var memberships []db.Membership

			BeforeEach(func() {
				memberships = []db.Membership{
					{ChannelID: "1"},
					{ChannelID: "2"},
					{ChannelID: "3"},
					{ChannelID: "4"},
				}

				dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).Return(memberships, nil)
			})

			Context("when the Users service fails to return some or all channels", func() {
				BeforeEach(func() {
					users.On("GetAll", mock.Anything, mock.Anything, mock.Anything).Return(nil, errors.New("🚨"))
				})

				It("returns Internal Server Error", func() {
					server.ServeHTTP(recorder, req)

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					Expect(recorder.Body.String()).To(ContainSubstring("failed to query channels for team"))
				})
			})

			Context("when the Users service returns some channels", func() {
				var channelsResult channels.ChannelsResult

				BeforeEach(func() {
					channelsResult = channels.ChannelsResult{
						Results: []channels.Channel{
							{
								ID:            1,
								PrimaryTeamID: 9000,
							},
							{
								ID:            2,
								PrimaryTeamID: 9000,
							},
							{
								ID:            3,
								PrimaryTeamID: 0,
							},
						},
					}

					channelIDs := []string{"1", "2", "3", "4"}
					users.On("GetAll", mock.Anything, channelIDs, mock.Anything).Return(&channelsResult, nil)
				})

				Context("when the Users service fails to delete at least one channel's primary team ID", func() {
					BeforeEach(func() {
						users.On("Set", mock.Anything, models.UpdateChannelProperties{
							ID:                  1,
							PrimaryTeamID:       nil,
							DeletePrimaryTeamID: true,
						}, mock.Anything).Return(nil).Once()

						users.On("Set", mock.Anything, models.UpdateChannelProperties{
							ID:                  2,
							PrimaryTeamID:       nil,
							DeletePrimaryTeamID: true,
						}, mock.Anything).Return(errors.New("failed 2")).Once()
					})

					It("returns Internal Server Error", func() {
						server.ServeHTTP(recorder, req)

						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
						Expect(recorder.Body.String()).To(ContainSubstring("failed to delete primary team ID for one or more channels"))
					})
				})

				Context("when the Users service successfully deletes EVERY channel's primary team ID", func() {
					BeforeEach(func() {
						users.On("Set", mock.Anything, models.UpdateChannelProperties{
							ID:                  1,
							PrimaryTeamID:       nil,
							DeletePrimaryTeamID: true,
						}, mock.Anything).Return(nil).Once()

						users.On("Set", mock.Anything, models.UpdateChannelProperties{
							ID:                  2,
							PrimaryTeamID:       nil,
							DeletePrimaryTeamID: true,
						}, mock.Anything).Return(nil).Once()
					})

					Context("when DB fails to delete the team", func() {
						BeforeEach(func() {
							dbWriter.On("DeleteTeam", mock.Anything, teamID).Return(errors.New("❌"))
						})

						It("returns Internal Server Error", func() {
							server.ServeHTTP(recorder, req)

							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
							Expect(recorder.Body.String()).To(ContainSubstring("failed to delete team 9000 and associated relations"))
						})
					})

					Context("when DB successfully deletes the team", func() {
						BeforeEach(func() {
							dbWriter.On("DeleteTeam", mock.Anything, teamID).Return(nil)
							cache.On("ClearTeam", mock.Anything, teamID).Return(nil)
							cache.On("ClearAllTeams", mock.Anything).Return(nil)
							cache.On("ClearChannelMemberships", mock.Anything, "1").Return(nil)
							cache.On("ClearChannelMemberships", mock.Anything, "2").Return(nil)
							cache.On("ClearChannelMemberships", mock.Anything, "3").Return(nil)
							cache.On("ClearChannelMemberships", mock.Anything, "4").Return(nil)
							cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
						})

						It("returns No Content", func() {
							server.ServeHTTP(recorder, req)

							Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
						})
					})
				})
			})
		})
	})
})
