package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strconv"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1ChannelInvitations", func() {
	var (
		dbReader  *mocks.DBReader
		users     *mocks.Users
		server    *Server
		recorder  *httptest.ResponseRecorder
		req       *http.Request
		err       error
		channelID string
	)

	BeforeEach(func() {
		channelID = "123"
		recorder = httptest.NewRecorder()
		dbReader = &mocks.DBReader{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			DBReader:         dbReader,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	Context("The channel ID is invalid", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/invitations", "invalid-channel-id")
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())
		})

		It("returns a 400", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
		})
	})

	Context("The channel does not exist", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/invitations", channelID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())

			users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(nil, &channels.ErrChannelNotFound{})
		})

		It("returns a 404", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusNotFound))
		})
	})

	Context("The channel lookup fails", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/invitations", channelID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())

			users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&channels.Channel{}, errors.New("some error"))
		})

		It("returns a 500", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
		})
	})

	Context("The channel exists", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/invitations", channelID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())

			channelIDInt, _ := strconv.Atoi(channelID)
			users.On("Get", mock.Anything, channelID, (*twitchclient.ReqOpts)(nil)).Return(&channels.Channel{
				ID: channelIDInt,
			}, nil)
		})

		Context("Getting the channel's invitations fails", func() {
			BeforeEach(func() {
				dbReader.On("GetInvitingTeamsForChannel", mock.Anything, channelID).
					Return([]db.Team{}, errors.New("stuff"))
			})

			It("returns a 500", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("The channel has been invited to some teams", func() {
			var teams []db.Team
			var logoID, logoURL string

			BeforeEach(func() {
				logoID = "logouid"
				logoURL = "logo.src"

				teams = []db.Team{
					{
						ID:          "123",
						Name:        "team_1",
						UserID:      "some-team-owner",
						DisplayName: "Team 1!",
						Logo: &db.Image{
							ID:  logoID,
							URL: logoURL,
						},
					},
					{
						ID:          "456",
						Name:        "team_2",
						UserID:      "other-team-owner",
						DisplayName: "Team 1!",
						Logo: &db.Image{
							ID:  logoID,
							URL: logoURL,
						},
					},
				}
				dbReader.On("GetInvitingTeamsForChannel", mock.Anything, channelID).Return(teams, nil)
			})

			It("returns a 200", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusOK))
			})

			It("returns the list of teams", func() {
				expectedResponseData := []v1.GetChannelInvitationsData{
					{
						Team: v1.Team{
							ID:          "123",
							Name:        "team_1",
							UserID:      "some-team-owner",
							DisplayName: "Team 1!",
							LogoID:      &logoID,
							LogoURL:     &logoURL,
						},
					},
					{
						Team: v1.Team{
							ID:          "456",
							Name:        "team_2",
							UserID:      "other-team-owner",
							DisplayName: "Team 1!",
							LogoID:      &logoID,
							LogoURL:     &logoURL,
						},
					},
				}

				server.ServeHTTP(recorder, req)

				response := v1.GetChannelInvitationsResponse{}
				err = json.Unmarshal(recorder.Body.Bytes(), &response)
				Expect(err).NotTo(HaveOccurred())

				Expect(response.Data).To(HaveLen(2))
				Expect(response.Data).To(Equal(expectedResponseData))
			})
		})

		Context("The channel has no pending invitations", func() {
			var teams []db.Team

			BeforeEach(func() {
				teams = []db.Team{}
				dbReader.On("GetInvitingTeamsForChannel", mock.Anything, channelID).Return(teams, nil)
			})

			It("returns a 200", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusOK))
			})

			It("returns an empty list", func() {
				server.ServeHTTP(recorder, req)

				response := v1.GetChannelInvitationsResponse{}
				err = json.Unmarshal(recorder.Body.Bytes(), &response)
				Expect(err).NotTo(HaveOccurred())

				Expect(response.Data).To(HaveLen(0))
			})
		})
	})
})
