package api

import (
	"context"
	"fmt"
	"net/http"
	"strconv"
	"time"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/client/channels"
)

func (s *Server) getV1ChannelMemberships(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	jsonWriter.Cacheable(10 * time.Second)

	ctx := req.Context()
	channelID := ctx.Value(contextKeyChannelID).(string)

	if rawResponse := s.fetchCachedChannelMemberships(ctx, channelID); rawResponse != nil {
		jsonWriter.OKRaw(rawResponse)
		return
	}

	user, err := s.users.Get(ctx, channelID, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.NotFound(fmt.Sprintf("channel with ID %s not found", channelID))
		default:
			jsonWriter.InternalServerError("users_service: failed to query channel", err)
		}
		return
	}

	dbMemberships, err := s.dbReader.GetChannelMemberships(ctx, channelID)
	if err != nil {
		jsonWriter.InternalServerError("db: failed to query for channel's memberships", err)
		return
	}

	response := v1.GetChannelMembershipsResponse{
		Data: toV1ChannelMemberships(dbMemberships, user.PrimaryTeamID),
	}

	go s.cacheChannelMemberships(context.Background(), channelID, response)

	jsonWriter.OK(response)
}

func toV1ChannelMemberships(memberships []db.Membership, primaryTeamID int) []v1.GetChannelMembershipsData {
	channelMemberships := make([]v1.GetChannelMembershipsData, len(memberships))

	hasPrimary := false
	for idx, membership := range memberships {
		primary := membership.Team.ID == strconv.Itoa(primaryTeamID)
		if primary {
			hasPrimary = true
		}
		channelMemberships[idx] = v1.GetChannelMembershipsData{
			Primary:         primary,
			StatsRevealed:   membership.StatsRevealed,
			RevenueRevealed: membership.RevenueRevealed,
			Team:            transformDBTeamToV1Team(*membership.Team),
		}
	}

	if !hasPrimary && len(channelMemberships) > 0 {
		channelMemberships[0].Primary = true
	}
	return channelMemberships
}
