package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	"strconv"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1ChannelMemberships", func() {
	var (
		mockedCache                *mocks.Cache
		dbReader                   *mocks.DBReader
		users                      *mocks.Users
		server                     *Server
		recorder                   *httptest.ResponseRecorder
		req                        *http.Request
		err                        error
		channelID                  string
		logoID, logoURL1, logoURL2 string
		expectedResponseData       []v1.GetChannelMembershipsData
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		mockedCache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			Cache:            mockedCache,
			DBReader:         dbReader,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		channelID = "123"
		logoID = "logouid"
		logoURL1 = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team_1-team_logo_image-logouid-600x600.jpeg"
		logoURL2 = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team_2-team_logo_image-logouid-600x600.jpeg"

		expectedResponseData = []v1.GetChannelMembershipsData{
			{
				StatsRevealed:   false,
				RevenueRevealed: true,
				Primary:         true,
				Team: v1.Team{
					ID:          "123",
					Name:        "team_1",
					UserID:      "some-team-owner",
					DisplayName: "Team 1!",
					LogoID:      &logoID,
					LogoURL:     &logoURL1,
				},
			},
			{
				StatsRevealed:   true,
				RevenueRevealed: false,
				Primary:         false,
				Team: v1.Team{
					ID:          "456",
					Name:        "team_2",
					UserID:      "other-team-owner",
					DisplayName: "Team 1!",
					LogoID:      &logoID,
					LogoURL:     &logoURL2,
				},
			},
		}
	})

	Context("When the channel ID is invalid", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/memberships", "invalid-channel-id")
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())
		})

		It("returns a 400", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
		})
	})

	Context("When cached", func() {
		BeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/memberships", channelID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())

			jsonResponse, _ := json.Marshal(v1.GetChannelMembershipsResponse{
				Data: expectedResponseData,
			})
			mockedCache.On("GetChannelMemberships", mock.Anything, channelID).Return(string(jsonResponse), nil)
		})

		AfterEach(func() {
			mockedCache.AssertExpectations(GinkgoT())
		})

		It("returns a 200", func() {
			server.ServeHTTP(recorder, req)

			response := v1.GetChannelMembershipsResponse{}
			err = json.Unmarshal(recorder.Body.Bytes(), &response)
			Expect(err).NotTo(HaveOccurred())

			Expect(recorder.Code).To(Equal(http.StatusOK))
			Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
			Expect(response.Data).To(HaveLen(2))
			Expect(response.Data).To(Equal(expectedResponseData))
		})
	})

	Context("When not cached", func() {
		BeforeEach(func() {
			mockedCache.On("GetChannelMemberships", mock.Anything, channelID).Return("", cache.ErrNoChannelMemberships)
		})

		Context("When the channel does not exist", func() {
			BeforeEach(func() {
				path := fmt.Sprintf("/v1/channels/%s/memberships", channelID)
				req, err = http.NewRequest(http.MethodGet, path, nil)
				Expect(err).NotTo(HaveOccurred())

				users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, &channels.ErrChannelNotFound{})
			})

			It("returns a 404", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusNotFound))
			})
		})

		Context("When the channel lookup fails", func() {
			BeforeEach(func() {
				path := fmt.Sprintf("/v1/channels/%s/memberships", channelID)
				req, err = http.NewRequest(http.MethodGet, path, nil)
				Expect(err).NotTo(HaveOccurred())

				users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, errors.New("some error"))
			})

			It("returns a 500", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("When the channel exists with a primary team", func() {
			BeforeEach(func() {
				path := fmt.Sprintf("/v1/channels/%s/memberships", channelID)
				req, err = http.NewRequest(http.MethodGet, path, nil)
				Expect(err).NotTo(HaveOccurred())

				channelIDInt, _ := strconv.Atoi(channelID)
				foundChannel := channels.Channel{
					ID:            channelIDInt,
					PrimaryTeamID: 123,
				}
				users.On("Get", mock.Anything, channelID, mock.Anything).Return(&foundChannel, nil)
			})

			Context("When getting the channel's memberships fails", func() {
				BeforeEach(func() {
					dbReader.On("GetChannelMemberships", mock.Anything, channelID).
						Return([]db.Membership{}, errors.New("stuff"))
				})

				It("returns a 500", func() {
					server.ServeHTTP(recorder, req)

					Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("When the channel is a member of some teams", func() {
				var teamMemberships []db.Membership
				var logoID, logoURL1, logoURL2 string

				BeforeEach(func() {
					logoID = "logouid"
					logoURL1 = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team_1-team_logo_image-logouid-600x600.jpeg"
					logoURL2 = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team_2-team_logo_image-logouid-600x600.jpeg"

					teamMemberships = []db.Membership{
						{
							StatsRevealed:   false,
							RevenueRevealed: true,
							ChannelID:       channelID,
							TeamID:          "123",
							Team: &db.Team{
								ID:          "123",
								Name:        "team_1",
								UserID:      "some-team-owner",
								DisplayName: "Team 1!",
								Logo: &db.Image{
									ID:  logoID,
									URL: logoURL1,
								},
							},
						},
						{
							StatsRevealed:   true,
							RevenueRevealed: false,
							ChannelID:       channelID,
							TeamID:          "456",
							Team: &db.Team{
								ID:          "456",
								Name:        "team_2",
								UserID:      "other-team-owner",
								DisplayName: "Team 1!",
								Logo: &db.Image{
									ID:  logoID,
									URL: logoURL2,
								},
							},
						},
					}
					dbReader.On("GetChannelMemberships", mock.Anything, channelID).Return(teamMemberships, nil)

					jsonResponse, _ := json.Marshal(v1.GetChannelMembershipsResponse{
						Data: expectedResponseData,
					})
					mockedCache.On("SetChannelMemberships", mock.Anything, channelID, string(jsonResponse)).Return(nil)
				})

				It("returns the list of teams", func() {
					server.ServeHTTP(recorder, req)

					response := v1.GetChannelMembershipsResponse{}
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(response.Data).To(HaveLen(2))
					Expect(response.Data).To(Equal(expectedResponseData))
				})
			})
		})

		Context("The channel exists without a primary team", func() {
			BeforeEach(func() {
				path := fmt.Sprintf("/v1/channels/%s/memberships", channelID)
				req, err = http.NewRequest(http.MethodGet, path, nil)
				Expect(err).NotTo(HaveOccurred())

				channelIDInt, _ := strconv.Atoi(channelID)
				foundChannel := channels.Channel{
					ID:            channelIDInt,
					PrimaryTeamID: 0,
				}
				users.On("Get", mock.Anything, channelID, mock.Anything).Return(&foundChannel, nil)

				jsonResponse, _ := json.Marshal(v1.GetChannelMembershipsResponse{
					Data: expectedResponseData,
				})
				mockedCache.On("SetChannelMemberships", mock.Anything, channelID, string(jsonResponse)).Return(nil)
			})

			Context("When the channel is a member of some teams", func() {
				var teamMemberships []db.Membership
				var logoID, logoURL1, logoURL2 string

				BeforeEach(func() {
					logoID = "logouid"
					logoURL1 = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team_1-team_logo_image-logouid-600x600.jpeg"
					logoURL2 = "https://static-cdn.jtvnw.net/jtv_user_pictures/team-team_2-team_logo_image-logouid-600x600.jpeg"

					teamMemberships = []db.Membership{
						{
							StatsRevealed:   false,
							RevenueRevealed: true,
							ChannelID:       channelID,
							TeamID:          "123",
							Team: &db.Team{
								ID:          "123",
								Name:        "team_1",
								UserID:      "some-team-owner",
								DisplayName: "Team 1!",
								Logo: &db.Image{
									ID:  logoID,
									URL: logoURL1,
								},
							},
						},
						{
							StatsRevealed:   true,
							RevenueRevealed: false,
							ChannelID:       channelID,
							TeamID:          "456",
							Team: &db.Team{
								ID:          "456",
								Name:        "team_2",
								UserID:      "other-team-owner",
								DisplayName: "Team 1!",
								Logo: &db.Image{
									ID:  logoID,
									URL: logoURL2,
								},
							},
						},
					}
					dbReader.On("GetChannelMemberships", mock.Anything, channelID).Return(teamMemberships, nil)

					jsonResponse, _ := json.Marshal(v1.GetChannelMembershipsResponse{
						Data: expectedResponseData,
					})
					mockedCache.On("SetChannelMemberships", mock.Anything, channelID, string(jsonResponse)).Return(nil)
				})

				It("returns the list of teams with the first team as primary", func() {
					server.ServeHTTP(recorder, req)
					Expect(recorder.Code).To(Equal(http.StatusOK))

					response := v1.GetChannelMembershipsResponse{}
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(response.Data).To(HaveLen(2))
					Expect(response.Data).To(Equal(expectedResponseData))
				})
			})
		})
	})
})
