package api

import (
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
)

// GET /v1/teams/:team_id/featured_channels
func (s *Server) getV1FeaturedChannels(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	jsonWriter.Cacheable(1 * time.Minute)

	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	_, err := s.dbReader.GetTeamByID(ctx, teamID)
	switch {
	case err == db.ErrNoTeam:
		jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		return
	case err != nil:
		jsonWriter.InternalServerError("db: failed to query team", err)
		return
	}

	featuredChannels, err := s.dbReader.GetFeaturedChannels(ctx, teamID)
	if err != nil {
		jsonWriter.InternalServerError("db: failed to query team featured channels", err)
		return
	}

	data := make([]v1.GetFeaturedChannelsData, len(featuredChannels))
	for idx, featuredChannel := range featuredChannels {
		data[idx] = v1.GetFeaturedChannelsData{
			ChannelID: featuredChannel.ChannelID,
			TeamID:    featuredChannel.TeamID,
		}
	}

	response := v1.GetFeaturedChannelsResponse{
		Data: data,
	}

	jsonWriter.OK(response)
}
