package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1FeaturedChannels", func() {
	var (
		dbReader *mocks.DBReader
		server   *Server
		recorder *httptest.ResponseRecorder
		req      *http.Request
		err      error
		teamID   string
	)

	BeforeEach(func() {
		teamID = "123"
		recorder = httptest.NewRecorder()
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	JustBeforeEach(func() {
		path := fmt.Sprintf("/v1/teams/%s/featured_channels", teamID)
		req, err = http.NewRequest(http.MethodGet, path, nil)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)
	})

	Context("Team id is invalid", func() {
		BeforeEach(func() {
			teamID = "not-a-valid-team-id"
		})

		It("Returns bad request", func() {
			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
		})
	})

	Context("Team is not in db", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
		})

		It("Returns not found", func() {
			Expect(recorder.Code).To(Equal(http.StatusNotFound))
		})
	})

	Context("Retrieving team by ID fails in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("failed to query"))
		})

		It("returns internal server error", func() {
			Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
		})
	})

	Context("Retrieving team by ID succeeds in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
				ID:     teamID,
				UserID: "owner",
			}, nil)
		})

		Context("Retrieving featured channels for team fails in db", func() {
			BeforeEach(func() {
				dbReader.On("GetFeaturedChannels", mock.Anything, teamID).Return(
					nil,
					errors.New("failed to query"),
				)
			})

			It("Returns internal server error", func() {
				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("Retrieving featured channels for team succeeds in db", func() {
			BeforeEach(func() {
				var returnedFeaturedChannels = []db.FeaturedChannel{
					{
						ChannelID: "some channel",
						TeamID:    teamID,
					},
				}

				dbReader.On("GetFeaturedChannels", mock.Anything, teamID).Return(returnedFeaturedChannels, nil)
			})

			It("Returns the featured channels", func() {
				response := v1.GetFeaturedChannelsResponse{}
				err = json.Unmarshal(recorder.Body.Bytes(), &response)
				Expect(err).NotTo(HaveOccurred())

				Expect(recorder.Code).To(Equal(http.StatusOK))
				Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=60"}))

				Expect(response.Data).To(HaveLen(1))
				Expect(response.Data).To(Equal([]v1.GetFeaturedChannelsData{
					{
						ChannelID: "some channel",
						TeamID:    teamID,
					},
				}))
			})
		})
	})
})
