package api

import (
	"fmt"
	"net/http"
	"strconv"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/mapping"
)

const (
	getLiveFeaturedChannelsDefaultLimit uint64 = 10
)

// GET /v1/teams/:team_id/live_featured_channels
func (s *Server) getV1LiveFeaturedChannels(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()

	limit := getLiveFeaturedChannelsDefaultLimit
	if limitQuery := req.URL.Query().Get("limit"); limitQuery != "" {
		parsed, err := strconv.ParseUint(limitQuery, 10, 64)
		if err != nil {
			jsonWriter.BadRequest(fmt.Sprintf("invalid limit: %s", err))
			return
		}
		if parsed <= 0 {
			jsonWriter.BadRequest("limit must be greater than 0")
			return
		}

		limit = parsed
	}

	teamID := ctx.Value(contextKeyTeamID).(string)
	_, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to query team", err)
		}
		return
	}

	featuredChannels, err := s.dbReader.GetFeaturedChannels(ctx, teamID)
	if err != nil {
		jsonWriter.InternalServerError("db: failed to query team featured channels", err)
		return
	}

	featuredChannelsMap := mapping.NewOrdered()

	for _, channel := range featuredChannels {
		featuredChannelsMap.Add(channel.ChannelID, channel)
	}

	liveIDs := s.liveline.GetStreamsByChannelIDs(ctx, featuredChannelsMap.Keys())

	liveFeaturedChannels, err := featuredChannelsMap.ValuesByKeys(liveIDs)
	if err != nil {
		jsonWriter.InternalServerError("failed to match liveline results to featured channels from db", err)
		return
	}

	data := make([]v1.GetLiveFeaturedChannelsData, 0, limit)
	count := uint64(0)
	for _, featuredChannelValue := range liveFeaturedChannels {
		featuredChannel := featuredChannelValue.(db.FeaturedChannel)
		data = append(data, v1.GetLiveFeaturedChannelsData{
			ChannelID: featuredChannel.ChannelID,
			TeamID:    featuredChannel.TeamID,
		})

		count++
		if count >= limit {
			break
		}
	}

	response := v1.GetLiveFeaturedChannelsResponse{
		Data: data,
	}

	jsonWriter.OK(response)
}
