package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1LiveFeaturedChannels", func() {
	var (
		livelineMock *mocks.Liveline
		dbReader     *mocks.DBReader
		server       *Server
		recorder     *httptest.ResponseRecorder
		req          *http.Request
		err          error
		teamID       string
	)

	BeforeEach(func() {
		teamID = "123"
		recorder = httptest.NewRecorder()
		dbReader = &mocks.DBReader{}
		livelineMock = &mocks.Liveline{}

		server = NewServer(&ServerParams{
			DBReader:         dbReader,
			Liveline:         livelineMock,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	JustBeforeEach(func() {
		path := fmt.Sprintf("/v1/teams/%s/live_featured_channels", teamID)
		req, err = http.NewRequest(http.MethodGet, path, nil)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)
	})

	Context("Team id is invalid", func() {
		BeforeEach(func() {
			teamID = "not-a-valid-team-id"
		})

		It("Returns bad request", func() {
			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
		})
	})

	Context("Team is not in db", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
		})

		It("Returns not found", func() {
			Expect(recorder.Code).To(Equal(http.StatusNotFound))
		})
	})

	Context("Retrieving team by ID fails in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("failed to query"))
		})

		It("returns internal server error", func() {
			Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
		})
	})

	Context("Retrieving team by ID succeeds in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
				ID:     teamID,
				UserID: "owner",
			}, nil)
		})

		Context("Retrieving featured channels for team fails in db", func() {
			BeforeEach(func() {
				dbReader.On("GetFeaturedChannels", mock.Anything, teamID).Return(
					nil,
					errors.New("failed to query"),
				)
			})

			It("Returns internal server error", func() {
				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("Retrieving featured channels for team succeeds in db", func() {
			var channelIDs []string

			BeforeEach(func() {
				channelIDs = []string{"456", "789"}
				var returnedFeaturedChannels = []db.FeaturedChannel{
					{
						ChannelID: channelIDs[0],
						TeamID:    teamID,
					}, {
						ChannelID: channelIDs[1],
						TeamID:    teamID,
					},
				}

				dbReader.On("GetFeaturedChannels", mock.Anything, teamID).Return(returnedFeaturedChannels, nil)
			})

			Context("Retrieving live featured channels returns no live channels in Liveline", func() {
				BeforeEach(func() {
					liveChannels := make([]string, 0)
					livelineMock.On("GetStreamsByChannelIDs", mock.Anything, channelIDs).Return(liveChannels)
				})

				It("Returns no live featured channels", func() {
					response := v1.GetLiveFeaturedChannelsResponse{}
					err = json.Unmarshal(recorder.Body.Bytes(), &response)

					Expect(err).NotTo(HaveOccurred())
					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(response.Data).To(HaveLen(0))
				})
			})

			Context("Retrieving live streams using memberships returns live channels in Liveline", func() {
				BeforeEach(func() {
					liveChannels := []string{channelIDs[1]}
					livelineMock.On("GetStreamsByChannelIDs", mock.Anything, channelIDs).Return(liveChannels)
				})

				It("Returns success", func() {
					response := v1.GetFeaturedChannelsResponse{}
					err = json.Unmarshal(recorder.Body.Bytes(), &response)

					Expect(err).NotTo(HaveOccurred())
					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(response.Data).To(HaveLen(1))
					Expect(response.Data).To(Equal([]v1.GetFeaturedChannelsData{
						{
							ChannelID: channelIDs[1],
							TeamID:    teamID,
						},
					}))
				})
			})
		})
	})
})
