package api

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
)

// GET /v1/teams/:team_id
func (s *Server) getV1Team(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	jsonWriter.Cacheable(10 * time.Second)

	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	if rawResponse := s.fetchCachedTeam(ctx, teamID); rawResponse != nil {
		jsonWriter.OKRaw(rawResponse)
		return
	}

	dbTeam, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("failed to query team from db", err)
		}
		return
	}

	response := v1.GetTeamResponse{
		Data: transformDBTeamToV1Team(dbTeam),
	}

	go s.cacheTeam(context.Background(), teamID, response)

	jsonWriter.OK(response)
}
