package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1TeamInvitations", func() {
	var (
		dbReader *mocks.DBReader
		server   *Server
		recorder *httptest.ResponseRecorder
		req      *http.Request
		err      error
		teamID   string
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
		teamID = "123"
	})

	Context("With no query params", func() {
		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/invitations", teamID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("Team is not in DB", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
			})

			It("Returns not found", func() {
				Expect(recorder.Code).To(Equal(http.StatusNotFound))
			})
		})

		Context("Retrieving team by ID fails in DB", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("failed to query"))
			})

			It("returns internal server error", func() {
				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("Retrieving team by ID succeeds in DB", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
					ID:     teamID,
					UserID: "owner",
				}, nil)
			})

			Context("Retrieving invitations for team fails in DB", func() {
				BeforeEach(func() {
					dbReader.On("GetTeamInvitations", mock.Anything, teamID).Return(
						nil,
						errors.New("failed to query"),
					)
				})

				It("Returns internal server error", func() {
					Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("Retrieving invitations for team succeeds in DB", func() {
				BeforeEach(func() {
					dbReader.On("GetTeamInvitations", mock.Anything, teamID).Return([]db.Invitation{
						{
							ChannelID: "some channel",
							TeamID:    teamID,
						},
					}, nil)
				})

				It("Returns success", func() {
					Expect(recorder.Code).To(Equal(http.StatusOK))
				})
			})
		})
	})
})
