package api

import (
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/mapping"
)

// GET /v1/teams/:team_id/live_memberships
func (s *Server) getV1TeamLiveMemberships(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	team, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("failed to query team from db", err)
		}
		return
	}

	memberships, err := s.dbReader.GetTeamMemberships(ctx, teamID, nil)
	if err != nil {
		jsonWriter.InternalServerError("failed to query memberships from db", err)
		return
	}

	membershipMap := mapping.NewOrdered()

	for _, membership := range memberships {
		membershipMap.Add(membership.ChannelID, membership)
	}

	liveIDs := s.liveline.GetStreamsByChannelIDs(ctx, membershipMap.Keys())

	liveMemberships, err := membershipMap.ValuesByKeys(liveIDs)
	if err != nil {
		jsonWriter.InternalServerError("failed to match liveline results to memberships from db", err)
		return
	}

	data := make([]v1.GetTeamLiveMembershipsData, len(liveMemberships))
	for idx, membershipValue := range liveMemberships {
		membership := membershipValue.(db.Membership)
		data[idx] = v1.GetTeamLiveMembershipsData{
			ChannelID:       membership.ChannelID,
			TeamID:          membership.TeamID,
			RevenueRevealed: membership.RevenueRevealed,
			StatsRevealed:   membership.StatsRevealed,
		}
	}

	response := v1.GetTeamLiveMembershipsResponse{
		Meta: v1.GetTeamLiveMembershipsMeta{
			Team: transformDBTeamToV1Team(team),
		},
		Data: data,
	}

	jsonWriter.OK(response)
}
