package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1TeamLiveMemberships", func() {
	var (
		dbReader     *mocks.DBReader
		livelineMock *mocks.Liveline
		server       *Server
		recorder     *httptest.ResponseRecorder
		req          *http.Request
		err          error
		teamID       string
	)

	BeforeEach(func() {
		teamID = "123"
		recorder = httptest.NewRecorder()
		livelineMock = &mocks.Liveline{}
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			Liveline:         livelineMock,
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	JustBeforeEach(func() {
		path := fmt.Sprintf("/v1/teams/%s/live_memberships", teamID)
		req, err = http.NewRequest(http.MethodGet, path, nil)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)
	})

	Context("Team is not in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
		})

		It("Returns not found", func() {
			Expect(recorder.Code).To(Equal(http.StatusNotFound))
		})
	})

	Context("Retrieving team by ID fails in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("failed to query"))
		})

		It("returns internal server error", func() {
			Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
		})
	})

	Context("Retrieving team by ID succeeds in DB", func() {
		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
				ID:     teamID,
				UserID: "owner",
			}, nil)
		})

		Context("Retrieving memberships for team fails in DB", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).
					Return(nil, errors.New("failed to query"))
			})

			It("Returns internal server error", func() {
				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("Retrieving memberships for team succeeds in DB", func() {
			var channelIDs []string

			BeforeEach(func() {
				channelIDs = []string{"123", "789"}
				dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).Return([]db.Membership{
					{
						ChannelID:     channelIDs[0],
						TeamID:        teamID,
						StatsRevealed: true,
					},
					{
						ChannelID:     channelIDs[1],
						TeamID:        teamID,
						StatsRevealed: true,
					},
				}, nil)
			})

			Context("Retrieving live streams using memberships returns no live channels in Liveline", func() {
				BeforeEach(func() {
					var liveChannels []string
					livelineMock.On("GetStreamsByChannelIDs", mock.Anything, channelIDs).Return(liveChannels)
				})

				It("Returns success", func() {
					Expect(recorder.Code).To(Equal(http.StatusOK))
				})
			})

			Context("Retrieving live streams using memberships returns live channels in Liveline", func() {
				BeforeEach(func() {
					liveChannels := channelIDs[1:2]
					livelineMock.On("GetStreamsByChannelIDs", mock.Anything, channelIDs).Return(liveChannels)
				})

				It("Returns success", func() {
					Expect(recorder.Code).To(Equal(http.StatusOK))
				})
			})
		})
	})
})
