package api

import (
	"context"
	"fmt"
	"net/http"
	"time"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/params"
)

// GET /v1/teams/:team_id/memberships
func (s *Server) getV1TeamMemberships(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	jsonWriter.Cacheable(10 * time.Second)

	ctx := req.Context()

	params, err := getV1TeamMembershipsParams(ctx, req)
	if err != nil {
		jsonWriter.BadRequest(err)
		return
	}

	cacheQuery := cache.TeamMembershipsQuery{
		ID:      params.TeamID,
		Revenue: params.RevenueRevealed,
		Stats:   params.StatsRevealed,
	}

	if rawResponse := s.fetchCachedTeamMemberships(ctx, cacheQuery); rawResponse != nil {
		jsonWriter.OKRaw(rawResponse)
		return
	}

	team, err := s.dbReader.GetTeamByID(ctx, params.TeamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", params.TeamID))
		default:
			jsonWriter.InternalServerError("failed to query team from db", err)
		}
		return
	}

	memberships, err := s.dbReader.GetTeamMemberships(ctx, params.TeamID, &db.MembershipsFilter{
		RevenueRevealed: params.RevenueRevealed,
		StatsRevealed:   params.StatsRevealed,
	})

	if err != nil {
		jsonWriter.InternalServerError("failed to query memberships from db", err)
		return
	}

	data := make([]v1.GetTeamMembershipsData, len(memberships))
	for idx, membership := range memberships {
		data[idx] = v1.GetTeamMembershipsData{
			ChannelID:       membership.ChannelID,
			TeamID:          membership.TeamID,
			RevenueRevealed: membership.RevenueRevealed,
			StatsRevealed:   membership.StatsRevealed,
		}
	}

	response := v1.GetTeamMembershipsResponse{
		Meta: v1.GetTeamMembershipsMeta{
			Team: transformDBTeamToV1Team(team),
		},
		Data: data,
	}

	go s.cacheTeamMemberships(context.Background(), cacheQuery, response)

	jsonWriter.OK(response)
}

func getV1TeamMembershipsParams(ctx context.Context, req *http.Request) (v1.GetTeamMembershipsRequestParams, error) {
	teamID := ctx.Value(contextKeyTeamID).(string)

	return v1.GetTeamMembershipsRequestParams{
		TeamID:          teamID,
		RevenueRevealed: params.QueryToBoolPointer(req.URL.Query().Get("revenue_revealed")),
		StatsRevealed:   params.QueryToBoolPointer(req.URL.Query().Get("stats_revealed")),
	}, nil
}
