package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1TeamMemberships", func() {
	var (
		mockedCache         *mocks.Cache
		dbReader            *mocks.DBReader
		server              *Server
		recorder            *httptest.ResponseRecorder
		req                 *http.Request
		err                 error
		teamID, teamOwnerID string
		expectedTeam        v1.Team
		expectedMemberships []v1.GetTeamMembershipsData
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		mockedCache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			Cache:            mockedCache,
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamID = "123"
		teamOwnerID = "owner"

		expectedTeam = v1.Team{
			ID:     teamID,
			UserID: teamOwnerID,
		}
	})

	Context("With no query params", func() {
		var expectedJSONResponse string

		BeforeEach(func() {
			expectedMemberships = []v1.GetTeamMembershipsData{
				{
					ChannelID:       "some channel",
					TeamID:          teamID,
					RevenueRevealed: false,
					StatsRevealed:   true,
				},
			}

			marshaled, _ := json.Marshal(v1.GetTeamMembershipsResponse{
				Meta: v1.GetTeamMembershipsMeta{
					Team: expectedTeam,
				},
				Data: expectedMemberships,
			})
			expectedJSONResponse = string(marshaled)
		})

		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/memberships", teamID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("When cached", func() {
			BeforeEach(func() {
				mockedCache.On("GetTeamMemberships", mock.Anything, cache.TeamMembershipsQuery{
					ID: teamID,
				}).Return(expectedJSONResponse, nil)
			})

			AfterEach(func() {
				mockedCache.AssertExpectations(GinkgoT())
			})

			It("Returns OK", func() {
				var response v1.GetTeamMembershipsResponse
				err = json.Unmarshal(recorder.Body.Bytes(), &response)
				Expect(err).NotTo(HaveOccurred())

				Expect(recorder.Code).To(Equal(http.StatusOK))
				Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
				Expect(response.Meta.Team).To(Equal(expectedTeam))
				Expect(response.Data).To(Equal(expectedMemberships))
			})
		})

		Context("When not cached", func() {
			BeforeEach(func() {
				mockedCache.On("GetTeamMemberships", mock.Anything, cache.TeamMembershipsQuery{
					ID: teamID,
				}).Return("", cache.ErrNoTeamMemberships)
			})

			Context("When the team is not in DB", func() {
				BeforeEach(func() {
					dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
				})

				It("Returns not found", func() {
					Expect(recorder.Code).To(Equal(http.StatusNotFound))
				})
			})

			Context("When retrieving team by ID fails in DB", func() {
				BeforeEach(func() {
					dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("failed to query"))
				})

				It("returns internal server error", func() {
					Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("When retrieving team by ID succeeds in DB", func() {
				BeforeEach(func() {
					dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
						ID:     teamID,
						UserID: "owner",
					}, nil)
				})

				Context("When retrieving memberships for team fails in DB", func() {
					BeforeEach(func() {
						dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).
							Return(nil, errors.New("failed to query"))
					})

					It("Returns internal server error", func() {
						Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
						Expect(recorder.Body.String()).To(ContainSubstring("failed to query memberships from db"))
					})
				})

				Context("When retrieving memberships for team succeeds in DB", func() {
					BeforeEach(func() {
						filter := &db.MembershipsFilter{}
						dbReader.On("GetTeamMemberships", mock.Anything, teamID, filter).Return([]db.Membership{
							{
								ChannelID:     "some channel",
								TeamID:        teamID,
								StatsRevealed: true,
							},
						}, nil)

						query := cache.TeamMembershipsQuery{
							ID: teamID,
						}
						mockedCache.On("SetTeamMemberships", mock.Anything, query, expectedJSONResponse).Return(nil)
					})

					It("Returns OK", func() {
						var response v1.GetTeamMembershipsResponse
						err = json.Unmarshal(recorder.Body.Bytes(), &response)
						Expect(err).NotTo(HaveOccurred())

						Expect(recorder.Code).To(Equal(http.StatusOK))
						Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
						Expect(response.Meta.Team).To(Equal(expectedTeam))
						Expect(response.Data).To(Equal(expectedMemberships))
					})
				})
			})
		})
	})

	Context("With a query param", func() {
		var (
			queryValue string
			queryParam string
		)

		BeforeEach(func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
				ID:     teamID,
				UserID: teamOwnerID,
			}, nil)
		})

		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/memberships", teamID)
			req, err = http.NewRequest(http.MethodGet, path, nil)
			Expect(err).NotTo(HaveOccurred())
			query := req.URL.Query()
			query.Set(queryParam, queryValue)
			req.URL.RawQuery = query.Encode()

			server.ServeHTTP(recorder, req)
		})

		Context("with 'stats_revealed' query param", func() {
			BeforeEach(func() {
				queryParam = "stats_revealed"
			})

			Context("set to true", func() {
				BeforeEach(func() {
					queryValue = "1"
					statsRevealed := true

					cacheQuery := cache.TeamMembershipsQuery{
						ID:    teamID,
						Stats: &statsRevealed,
					}

					mockedCache.On("GetTeamMemberships", mock.Anything, cacheQuery).Return("", cache.ErrNoTeamMemberships)

					filter := &db.MembershipsFilter{
						StatsRevealed: &statsRevealed,
					}
					dbReader.On("GetTeamMemberships", mock.Anything, teamID, filter).Return([]db.Membership{
						{
							ChannelID:     "some channel",
							TeamID:        teamID,
							StatsRevealed: statsRevealed,
						},
					}, nil)

					expectedMemberships = []v1.GetTeamMembershipsData{
						{
							ChannelID:     "some channel",
							TeamID:        teamID,
							StatsRevealed: statsRevealed,
						},
					}

					marshaled, _ := json.Marshal(v1.GetTeamMembershipsResponse{
						Meta: v1.GetTeamMembershipsMeta{
							Team: expectedTeam,
						},
						Data: expectedMemberships,
					})

					mockedCache.On("SetTeamMemberships", mock.Anything, cacheQuery, string(marshaled)).Return(nil)
				})

				It("returns OK", func() {
					var response v1.GetTeamMembershipsResponse
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(response.Meta.Team).To(Equal(expectedTeam))
					Expect(response.Data).To(Equal(expectedMemberships))
				})
			})

			Context("set to false", func() {
				BeforeEach(func() {
					queryValue = "0"
					statsRevealed := false

					cacheQuery := cache.TeamMembershipsQuery{
						ID:    teamID,
						Stats: &statsRevealed,
					}

					mockedCache.On("GetTeamMemberships", mock.Anything, cacheQuery).Return("", cache.ErrNoTeamMemberships)

					filter := &db.MembershipsFilter{
						StatsRevealed: &statsRevealed,
					}
					dbReader.On("GetTeamMemberships", mock.Anything, teamID, filter).Return([]db.Membership{
						{
							ChannelID:     "some channel",
							TeamID:        teamID,
							StatsRevealed: statsRevealed,
						},
					}, nil)

					expectedMemberships = []v1.GetTeamMembershipsData{
						{
							ChannelID:     "some channel",
							TeamID:        teamID,
							StatsRevealed: statsRevealed,
						},
					}

					marshaled, _ := json.Marshal(v1.GetTeamMembershipsResponse{
						Meta: v1.GetTeamMembershipsMeta{
							Team: expectedTeam,
						},
						Data: expectedMemberships,
					})

					mockedCache.On("SetTeamMemberships", mock.Anything, cacheQuery, string(marshaled)).Return(nil)
				})

				It("returns OK", func() {
					var response v1.GetTeamMembershipsResponse
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(response.Meta.Team).To(Equal(expectedTeam))
					Expect(response.Data).To(Equal(expectedMemberships))
				})
			})
		})

		Context("With 'revenue_revealed' query param", func() {
			BeforeEach(func() {
				queryParam = "revenue_revealed"
			})

			Context("set to true", func() {
				BeforeEach(func() {
					queryValue = "1"
					revenueRevealed := true

					cacheQuery := cache.TeamMembershipsQuery{
						ID:      teamID,
						Revenue: &revenueRevealed,
					}

					mockedCache.On("GetTeamMemberships", mock.Anything, cacheQuery).Return("", cache.ErrNoTeamMemberships)

					filter := &db.MembershipsFilter{
						RevenueRevealed: &revenueRevealed,
					}
					dbReader.On("GetTeamMemberships", mock.Anything, teamID, filter).Return([]db.Membership{
						{
							ChannelID:       "some channel",
							TeamID:          teamID,
							RevenueRevealed: revenueRevealed,
						},
					}, nil)

					expectedMemberships = []v1.GetTeamMembershipsData{
						{
							ChannelID:       "some channel",
							TeamID:          teamID,
							RevenueRevealed: revenueRevealed,
						},
					}

					marshaled, _ := json.Marshal(v1.GetTeamMembershipsResponse{
						Meta: v1.GetTeamMembershipsMeta{
							Team: expectedTeam,
						},
						Data: expectedMemberships,
					})

					mockedCache.On("SetTeamMemberships", mock.Anything, cacheQuery, string(marshaled)).Return(nil)
				})

				It("returns OK", func() {
					var response v1.GetTeamMembershipsResponse
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(response.Meta.Team).To(Equal(expectedTeam))
					Expect(response.Data).To(Equal(expectedMemberships))
				})
			})

			Context("set to false", func() {
				BeforeEach(func() {
					queryValue = "0"
					revenueRevealed := false

					cacheQuery := cache.TeamMembershipsQuery{
						ID:      teamID,
						Revenue: &revenueRevealed,
					}

					mockedCache.On("GetTeamMemberships", mock.Anything, cacheQuery).Return("", cache.ErrNoTeamMemberships)

					filter := &db.MembershipsFilter{
						RevenueRevealed: &revenueRevealed,
					}
					dbReader.On("GetTeamMemberships", mock.Anything, teamID, filter).Return([]db.Membership{
						{
							ChannelID:       "some channel",
							TeamID:          teamID,
							RevenueRevealed: revenueRevealed,
						},
					}, nil)

					expectedMemberships = []v1.GetTeamMembershipsData{
						{
							ChannelID:       "some channel",
							TeamID:          teamID,
							RevenueRevealed: revenueRevealed,
						},
					}

					marshaled, _ := json.Marshal(v1.GetTeamMembershipsResponse{
						Meta: v1.GetTeamMembershipsMeta{
							Team: expectedTeam,
						},
						Data: expectedMemberships,
					})

					mockedCache.On("SetTeamMemberships", mock.Anything, cacheQuery, string(marshaled)).Return(nil)
				})

				It("returns OK", func() {
					var response v1.GetTeamMembershipsResponse
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(response.Meta.Team).To(Equal(expectedTeam))
					Expect(response.Data).To(Equal(expectedMemberships))
				})
			})
		})
	})
})
