package api

import (
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"time"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1Team", func() {
	var (
		mockedCache *mocks.Cache
		dbReader    *mocks.DBReader
		server      *Server
		recorder    *httptest.ResponseRecorder
		req         *http.Request
		err         error

		id, name, displayName, userID, descriptionHTML                    string
		logoID, logoURL, bannerID, bannerURL, backgroundID, backgroundURL string
		createdAt, updatedAt                                              time.Time
		expectedTeam                                                      v1.Team
		expectedJSONResponse                                              string
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		mockedCache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			Cache:            mockedCache,
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		id = "123"
		name = "it_your_boy"
		displayName = "It Your Boy"
		userID = "9999"
		descriptionHTML = "<em>Emphasized!</em>"
		logoID = "logo_id"
		logoURL = "logo.src"
		bannerID = "banner_id"
		bannerURL = "banner.src"
		backgroundID = "background_id"
		backgroundURL = "background.src"
		createdAt = time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
		updatedAt = time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)

		expectedTeam = v1.Team{
			ID:                  id,
			Name:                name,
			UserID:              userID,
			DisplayName:         displayName,
			DescriptionHTML:     descriptionHTML,
			DescriptionMarkdown: "_Emphasized!_",
			LogoID:              &logoID,
			LogoURL:             &logoURL,
			BannerID:            &bannerID,
			BannerURL:           &bannerURL,
			BackgroundImageID:   &backgroundID,
			BackgroundImageURL:  &backgroundURL,
			CreatedAt:           &createdAt,
			UpdatedAt:           &updatedAt,
		}

		var jsonResponse []byte
		jsonResponse, err = json.Marshal(v1.GetTeamResponse{
			Data: expectedTeam,
		})
		Expect(err).NotTo(HaveOccurred())

		expectedJSONResponse = string(jsonResponse)
	})

	JustBeforeEach(func() {
		path := fmt.Sprintf("/v1/teams/%s", id)
		req, err = http.NewRequest(http.MethodGet, path, nil)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)
	})

	Context("When the team is in the cache", func() {
		BeforeEach(func() {
			mockedCache.On("GetTeam", mock.Anything, id).Return(expectedJSONResponse, nil)
		})

		AfterEach(func() {
			mockedCache.AssertExpectations(GinkgoT())
		})

		It("Returns OK", func() {
			response := v1.GetTeamResponse{}
			err = json.Unmarshal(recorder.Body.Bytes(), &response)
			Expect(err).NotTo(HaveOccurred())

			Expect(recorder.Code).To(Equal(http.StatusOK))
			Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))

			Expect(response.Data).To(Equal(expectedTeam))
		})
	})

	Context("When the team is not in the cache", func() {
		BeforeEach(func() {
			mockedCache.On("GetTeam", mock.Anything, id).Return("", cache.ErrNoTeam)
		})

		AfterEach(func() {
			dbReader.AssertExpectations(GinkgoT())
		})

		Context("When the team is not in DB", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, id).Return(db.Team{}, db.ErrNoTeam)
			})

			It("Returns Not Found", func() {
				Expect(recorder.Code).To(Equal(http.StatusNotFound))
			})
		})

		Context("When retrieving the team by ID fails in DB", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, id).Return(db.Team{}, errors.New("💩"))
			})

			It("returns Internal Server Error", func() {
				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("When retrieving the team by ID succeeds in DB", func() {
			BeforeEach(func() {
				team := db.Team{
					ID:          id,
					Name:        name,
					UserID:      userID,
					DisplayName: displayName,
					Logo: &db.Image{
						ID:  logoID,
						URL: logoURL,
					},
					Banner: &db.Image{
						ID:  bannerID,
						URL: bannerURL,
					},
					Background: &db.Image{
						ID:  backgroundID,
						URL: backgroundURL,
					},
					CreatedAt: &createdAt,
					UpdatedAt: &updatedAt,
				}
				team.SetDescriptionWithHTML(descriptionHTML)

				dbReader.On("GetTeamByID", mock.Anything, id).Return(team, nil)
				mockedCache.On("SetTeam", mock.Anything, id, expectedJSONResponse).Return(nil)
			})

			It("Returns OK", func() {
				var response v1.GetTeamResponse
				err = json.Unmarshal(recorder.Body.Bytes(), &response)
				Expect(err).NotTo(HaveOccurred())

				Expect(recorder.Code).To(Equal(http.StatusOK))
				Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))

				Expect(response.Data).To(Equal(expectedTeam))
			})
		})
	})
})
