package api

import (
	"context"
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strconv"
	"strings"
	"time"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"golang.org/x/sync/errgroup"
)

// GET /v1/teams
//
// Query parameters:
// - limit
// - offset
// - name
// - sort
// - user_id
func (s *Server) getV1Teams(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	jsonWriter.Cacheable(10 * time.Second)

	filter, err := toDBTeamsFilter(req.URL.Query())
	if err != nil {
		jsonWriter.BadRequest(err.Error())
		return
	}

	cacheQuery := cache.TeamsQuery(filter)

	if rawResponse := s.fetchCachedTeams(req.Context(), cacheQuery); rawResponse != nil {
		jsonWriter.OKRaw(rawResponse)
		return
	}

	group, ctx := errgroup.WithContext(req.Context())

	var total uint
	var teams []db.Team

	group.Go(func() error {
		var countError error
		total, countError = s.dbReader.GetTeamsCount(ctx, filter)
		return countError
	})

	group.Go(func() error {
		var teamsError error
		teams, teamsError = s.dbReader.GetTeams(ctx, filter)
		return teamsError
	})

	if err = group.Wait(); err != nil {
		jsonWriter.InternalServerError("failed to query teams from db", err)
		return
	}

	response := v1.GetTeamsResponse{
		Meta: v1.GetTeamsMeta{
			Limit:  *filter.Limit,
			Offset: filter.Offset,
			Sort:   v1.Sort(strings.ToLower(filter.OrderByDirection)),
			Total:  total,
		},
		Data: toV1Teams(teams),
	}

	go s.cacheTeams(context.Background(), cacheQuery, response)

	jsonWriter.OK(response)
}

func toDBTeamsFilter(query url.Values) (db.TeamsFilter, error) {
	limit := uint(db.GetTeamsDefaultLimit)
	offset := uint(0)
	direction := db.OrderByDirectionAsc

	if limitQuery := query.Get("limit"); limitQuery != "" {
		parsed, err := strconv.ParseUint(limitQuery, 10, 64)
		if err != nil {
			return db.TeamsFilter{}, errors.New("invalid limit")
		}

		if parsed > db.GetTeamsMaxLimit {
			return db.TeamsFilter{}, fmt.Errorf("limit cannot be greater than %d", db.GetTeamsMaxLimit)
		}

		limit = uint(parsed)
	}

	if offsetQuery := query.Get("offset"); offsetQuery != "" {
		parsed, err := strconv.ParseUint(offsetQuery, 10, 64)
		if err != nil {
			return db.TeamsFilter{}, errors.New("invalid offset")
		}

		offset = uint(parsed)
	}

	if sortQuery := query.Get("sort"); sortQuery != "" {
		direction = strings.ToUpper(sortQuery)

		if direction != db.OrderByDirectionAsc && direction != db.OrderByDirectionDesc {
			return db.TeamsFilter{}, fmt.Errorf("sort must be one of: %s, %s", v1.SortAsc, v1.SortDesc)
		}
	}

	return db.TeamsFilter{
		Limit:            &limit,
		Name:             query.Get("name"),
		Offset:           offset,
		OrderByDirection: direction,
		UserID:           query.Get("user_id"),
	}, nil
}

func toV1Teams(dbTeams []db.Team) []v1.Team {
	v1Teams := make([]v1.Team, len(dbTeams))

	for idx, team := range dbTeams {
		v1Teams[idx] = transformDBTeamToV1Team(team)
	}

	return v1Teams
}
