package api

import (
	"encoding/json"
	"errors"
	"net/http"
	"net/http/httptest"
	"time"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/cache"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("GetV1Teams", func() {
	var (
		mockedCache *mocks.Cache
		dbReader    *mocks.DBReader
		server      *Server
		recorder    *httptest.ResponseRecorder

		teamName                                                          string
		teamOwnerID                                                       string
		logoID, logoURL, bannerID, bannerURL, backgroundID, backgroundURL string
		expectedTeam                                                      v1.Team
		expectedJSONResponse                                              string
	)

	var (
		creationTime = time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
		updateTime   = time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		mockedCache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			Cache:            mockedCache,
			DBReader:         dbReader,
			Users:            &mocks.Users{},
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamName = "staff"
		teamOwnerID = "emmett"
		logoID = "logouid"
		logoURL = "logo.src"
		bannerID = "banneruid"
		bannerURL = "banner.src"
		backgroundID = "backgrounduid"
		backgroundURL = "background.src"
		createdAt := creationTime
		updatedAt := updateTime

		expectedTeam = v1.Team{
			ID:                  "some team id",
			Name:                teamName,
			UserID:              teamOwnerID,
			DisplayName:         "Team 1!",
			DescriptionHTML:     "<b>Bold team</b>",
			DescriptionMarkdown: "**Bold team**",
			LogoID:              &logoID,
			LogoURL:             &logoURL,
			BannerID:            &bannerID,
			BannerURL:           &bannerURL,
			BackgroundImageID:   &backgroundID,
			BackgroundImageURL:  &backgroundURL,
			CreatedAt:           &createdAt,
			UpdatedAt:           &updatedAt,
		}
	})

	Context("With query params", func() {
		var (
			req          *http.Request
			limit        uint
			limitString  string
			offset       uint
			offsetString string
			sort         string
			err          error
		)

		Context("with invalid limit", func() {
			BeforeEach(func() {
				limitString = "abc"

				req, err = http.NewRequest(http.MethodGet, "/v1/teams", nil)
				Expect(err).NotTo(HaveOccurred())

				query := req.URL.Query()
				query.Set("limit", limitString)
				req.URL.RawQuery = query.Encode()
			})

			It("returns 400", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusBadRequest))
				Expect(recorder.Body.String()).To(ContainSubstring("invalid limit"))
			})
		})

		Context("with limit too large", func() {
			BeforeEach(func() {
				limitString = "101"

				req, err = http.NewRequest(http.MethodGet, "/v1/teams", nil)
				Expect(err).NotTo(HaveOccurred())

				query := req.URL.Query()
				query.Set("limit", limitString)
				req.URL.RawQuery = query.Encode()
			})

			It("returns 400", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusBadRequest))
				Expect(recorder.Body.String()).To(ContainSubstring("limit cannot be greater than"))
			})
		})

		Context("with invalid offset", func() {
			BeforeEach(func() {
				offsetString = "abc"

				req, err = http.NewRequest(http.MethodGet, "/v1/teams", nil)
				Expect(err).NotTo(HaveOccurred())

				query := req.URL.Query()
				query.Set("offset", offsetString)
				req.URL.RawQuery = query.Encode()
			})

			It("returns 400", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusBadRequest))
				Expect(recorder.Body.String()).To(ContainSubstring("invalid offset"))
			})
		})

		Context("with invalid sort", func() {
			BeforeEach(func() {
				sort = "abc"

				req, err = http.NewRequest(http.MethodGet, "/v1/teams", nil)
				Expect(err).NotTo(HaveOccurred())

				query := req.URL.Query()
				query.Set("sort", sort)
				req.URL.RawQuery = query.Encode()
			})

			It("returns 400", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusBadRequest))
				Expect(recorder.Body.String()).To(ContainSubstring("sort must be one of: asc, desc"))
			})
		})

		Context("with valid limit, offset, and sort", func() {
			var (
				total      uint
				cacheQuery cache.TeamsQuery
			)

			BeforeEach(func() {
				limit = 9
				limitString = "9"
				offset = 2
				offsetString = "2"
				sort = "desc"
				total = 333

				req, err = http.NewRequest(http.MethodGet, "/v1/teams", nil)
				Expect(err).NotTo(HaveOccurred())

				query := req.URL.Query()
				query.Set("limit", limitString)
				query.Set("name", teamName)
				query.Set("offset", offsetString)
				query.Set("sort", sort)
				query.Set("user_id", teamOwnerID)
				req.URL.RawQuery = query.Encode()

				var marshaled []byte
				marshaled, err = json.Marshal(v1.GetTeamsResponse{
					Meta: v1.GetTeamsMeta{
						Limit:  limit,
						Offset: offset,
						Sort:   v1.Sort(sort),
						Total:  total,
					},
					Data: []v1.Team{
						expectedTeam,
					},
				})
				Expect(err).NotTo(HaveOccurred())

				expectedJSONResponse = string(marshaled)

				cacheQuery = cache.TeamsQuery{
					Limit:            &limit,
					Name:             teamName,
					Offset:           2,
					OrderByDirection: db.OrderByDirectionDesc,
					UserID:           teamOwnerID,
				}
			})

			Context("when cached", func() {
				BeforeEach(func() {
					mockedCache.On("GetTeams", mock.Anything, cacheQuery).Return(expectedJSONResponse, nil)
				})

				It("returns OK", func() {
					server.ServeHTTP(recorder, req)

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(recorder.Body.String()).To(Equal(expectedJSONResponse))
				})
			})

			Context("when not cached", func() {
				BeforeEach(func() {
					mockedCache.On("GetTeams", mock.Anything, cacheQuery).Return("", cache.ErrNoTeams)
				})

				Context("when DB fails to select the total team count", func() {
					It("Returns 500", func() {
						dbReader.On("GetTeamsCount", mock.Anything, mock.Anything).Return(uint(0), errors.New("failed to query"))
						dbReader.On("GetTeams", mock.Anything, mock.Anything).Return([]db.Team{}, nil)

						server.ServeHTTP(recorder, req)

						Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
					})
				})

				Context("when DB fails to select teams", func() {
					It("Returns 500", func() {
						dbReader.On("GetTeamsCount", mock.Anything, mock.Anything).Return(total, nil)
						dbReader.On("GetTeams", mock.Anything, mock.Anything).Return([]db.Team{}, errors.New("failed to query"))

						server.ServeHTTP(recorder, req)

						Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
					})
				})

				Context("when DB succeeds", func() {
					BeforeEach(func() {
						createdAt := creationTime
						updatedAt := updateTime

						team := db.Team{
							ID:          "some team id",
							Name:        teamName,
							UserID:      teamOwnerID,
							DisplayName: "Team 1!",
							Logo: &db.Image{
								ID:  logoID,
								URL: logoURL,
							},
							Banner: &db.Image{
								ID:  bannerID,
								URL: bannerURL,
							},
							Background: &db.Image{
								ID:  backgroundID,
								URL: backgroundURL,
							},
							CreatedAt: &createdAt,
							UpdatedAt: &updatedAt,
						}
						team.SetDescriptionWithHTML("<b>Bold team</b>")

						dbReader.On("GetTeams", mock.Anything, db.TeamsFilter{
							Limit:            &limit,
							Name:             teamName,
							Offset:           2,
							OrderByDirection: db.OrderByDirectionDesc,
							UserID:           teamOwnerID,
						}).Return([]db.Team{team}, nil)

						dbReader.On("GetTeamsCount", mock.Anything, mock.Anything).Return(total, nil)
						mockedCache.On("SetTeams", mock.Anything, cacheQuery, expectedJSONResponse).Return(nil)
					})

					It("Returns found team when it is found", func() {
						server.ServeHTTP(recorder, req)

						response := v1.GetTeamsResponse{}
						err = json.Unmarshal(recorder.Body.Bytes(), &response)
						Expect(err).NotTo(HaveOccurred())

						Expect(recorder.Code).To(Equal(http.StatusOK))
						Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
						Expect(response.Data).To(HaveLen(1))
						Expect(response.Data[0]).To(Equal(expectedTeam))
					})
				})
			})
		})
	})

	Context("Without query params", func() {
		var (
			req        *http.Request
			err        error
			limit      uint
			offset     uint
			sort       v1.Sort
			total      uint
			cacheQuery cache.TeamsQuery
		)

		BeforeEach(func() {
			req, err = http.NewRequest(http.MethodGet, "/v1/teams", nil)
			Expect(err).NotTo(HaveOccurred())

			limit = db.GetTeamsDefaultLimit
			offset = 0
			sort = v1.SortAsc
			total = 333

			var marshaled []byte
			marshaled, err = json.Marshal(v1.GetTeamsResponse{
				Meta: v1.GetTeamsMeta{
					Limit:  limit,
					Offset: offset,
					Sort:   sort,
					Total:  total,
				},
				Data: []v1.Team{
					expectedTeam,
				},
			})
			Expect(err).NotTo(HaveOccurred())

			expectedJSONResponse = string(marshaled)

			cacheQuery = cache.TeamsQuery{
				Limit:            &limit,
				Offset:           offset,
				OrderByDirection: db.OrderByDirectionAsc,
			}
		})

		Context("when cached", func() {
			BeforeEach(func() {
				mockedCache.On("GetTeams", mock.Anything, cacheQuery).Return(expectedJSONResponse, nil)
			})

			It("returns OK", func() {
				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusOK))
				Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
				Expect(recorder.Body.String()).To(Equal(expectedJSONResponse))
			})
		})

		Context("when not cached", func() {
			BeforeEach(func() {
				mockedCache.On("GetTeams", mock.Anything, cacheQuery).Return("", cache.ErrNoTeams)
			})

			Context("when DB fails to select the total team count", func() {
				It("Returns 500", func() {
					dbReader.On("GetTeamsCount", mock.Anything, mock.Anything).Return(uint(0), errors.New("failed to query"))
					dbReader.On("GetTeams", mock.Anything, mock.Anything).Return([]db.Team{}, nil)

					server.ServeHTTP(recorder, req)

					Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("when DB fails to select teams", func() {
				It("Returns 500", func() {
					dbReader.On("GetTeamsCount", mock.Anything, mock.Anything).Return(total, nil)
					dbReader.On("GetTeams", mock.Anything, mock.Anything).Return([]db.Team{}, errors.New("failed to query"))

					server.ServeHTTP(recorder, req)

					Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("when DB succeeds", func() {
				BeforeEach(func() {
					createdAt := creationTime
					updatedAt := updateTime

					team := db.Team{
						ID:          "some team id",
						Name:        teamName,
						UserID:      teamOwnerID,
						DisplayName: "Team 1!",
						Logo: &db.Image{
							ID:  logoID,
							URL: logoURL,
						},
						Banner: &db.Image{
							ID:  bannerID,
							URL: bannerURL,
						},
						Background: &db.Image{
							ID:  backgroundID,
							URL: backgroundURL,
						},
						CreatedAt: &createdAt,
						UpdatedAt: &updatedAt,
					}
					team.SetDescriptionWithHTML("<b>Bold team</b>")

					filter := db.TeamsFilter{
						Limit:            &limit,
						OrderByDirection: db.OrderByDirectionAsc,
					}

					dbReader.On("GetTeams", mock.Anything, filter).Return([]db.Team{team}, nil)
					dbReader.On("GetTeamsCount", mock.Anything, filter).Return(total, nil)
					mockedCache.On("SetTeams", mock.Anything, cacheQuery, expectedJSONResponse).Return(nil)
				})

				It("returns OK", func() {
					server.ServeHTTP(recorder, req)

					response := v1.GetTeamsResponse{}
					err = json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(recorder.Code).To(Equal(http.StatusOK))
					Expect(recorder.Header()["Cache-Control"]).To(Equal([]string{"public", "max-age=10"}))
					Expect(response.Meta.Limit).To(Equal(uint(db.GetTeamsDefaultLimit)))
					Expect(response.Meta.Offset).To(Equal(offset))
					Expect(response.Meta.Sort).To(Equal(v1.SortAsc))
					Expect(response.Meta.Total).To(Equal(total))
					Expect(response.Data).To(HaveLen(1))
					Expect(response.Data[0]).To(Equal(expectedTeam))
				})
			})
		})
	})
})
