package api

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/description"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/image"
	"code.justin.tv/cb/roster/internal/name"
	"code.justin.tv/cb/roster/internal/s3"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

func (s *Server) patchV1Team(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	body, err := decodePatchV1TeamRequestBody(req.Body)
	if err != nil {
		jsonWriter.BadRequest(fmt.Sprintf("invalid request body: %s", err))
		return
	}

	team, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to find team", err)
		}
		return
	}

	if body.UserID != nil && *body.UserID != team.UserID {
		_, err = s.users.GetByIDAndParams(ctx, *body.UserID, &models.ChannelFilterParams{
			NotDeleted:      true,
			NoTOSViolation:  true,
			NoDMCAViolation: true,
		}, nil)
		if err != nil {
			switch err.(type) {
			case *channels.ErrChannelNotFound:
				jsonWriter.UnprocessableEntity("users service: user not found")
			default:
				jsonWriter.InternalServerError("users service: failed to look up channel", err)
			}
			return
		}

		team.UserID = *body.UserID
	}

	if shouldChangeImage(team.Logo, body.LogoID) {
		logo, found, s3Error := s.findAndSaveImage(ctx, team, *body.LogoID.Value, image.CategoryLogo)
		if s3Error != nil {
			jsonWriter.InternalServerError("s3: failed to find logo", s3Error)
			return
		}
		if !found {
			jsonWriter.UnprocessableEntity(fmt.Sprintf("logo with id %s not found or invalid", *body.LogoID.Value))
			return
		}

		team.SetLogo(logo.ID, logo.Format)
	} else if shouldRemoveImage(body.LogoID) {
		team.Logo = nil
	}

	if shouldChangeImage(team.Banner, body.BannerID) {
		banner, found, s3Error := s.findAndSaveImage(ctx, team, *body.BannerID.Value, image.CategoryBanner)
		if s3Error != nil {
			jsonWriter.InternalServerError("s3: failed to find banner", s3Error)
			return
		}
		if !found {
			jsonWriter.UnprocessableEntity(fmt.Sprintf("banner with id %s not found or invalid", *body.BannerID.Value))
			return
		}

		team.SetBanner(banner.ID, banner.Format)
	} else if shouldRemoveImage(body.BannerID) {
		team.Banner = nil
	}

	if shouldChangeImage(team.Background, body.BackgroundImageID) {
		background, found, s3Error := s.findAndSaveImage(ctx, team, *body.BackgroundImageID.Value, image.CategoryBackground)
		if s3Error != nil {
			jsonWriter.InternalServerError("s3: failed to find background image", s3Error)
			return
		}
		if !found {
			jsonWriter.UnprocessableEntity(fmt.Sprintf("background with id %s not found or invalid", *body.BackgroundImageID.Value))
			return
		}

		team.SetBackground(background.ID, background.Format)
	} else if shouldRemoveImage(body.BackgroundImageID) {
		team.Background = nil
	}

	if body.DisplayName != nil {
		team.DisplayName = *body.DisplayName
	}

	if body.DescriptionMarkdown != nil {
		team.SetDescriptionWithMarkdown(*body.DescriptionMarkdown)
	}

	err = s.dbWriter.UpdateTeam(ctx, team)
	if err != nil {
		switch err {
		case db.ErrNoTeamForUpdate:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to update team", err)
		}
		return
	}

	go s.expireCachedTeam(context.Background(), teamID)
	go s.expireCachedTeams(context.Background())
	go s.expireCachedChannelMembershipsByTeam(context.Background(), teamID)

	jsonWriter.OK(v1.PatchTeamResponse{
		Data: transformDBTeamToV1Team(team),
	})
}

// This representation of the PATCH request body differs from api/v1
// because the application server needs to distinguish the differences among:
// - Image ID not present
// - Image ID present but "null"
// - Image ID present with string value
type patchV1TeamRequestBody struct {
	DisplayName         *string       `json:"display_name"`
	UserID              *string       `json:"user_id"`
	DescriptionMarkdown *string       `json:"description_markdown"`
	LogoID              v1.NullString `json:"logo_id"`
	BannerID            v1.NullString `json:"banner_id"`
	BackgroundImageID   v1.NullString `json:"background_image_id"`
}

func decodePatchV1TeamRequestBody(reqBody io.ReadCloser) (patchV1TeamRequestBody, error) {
	var decoded patchV1TeamRequestBody

	err := json.NewDecoder(reqBody).Decode(&decoded)
	if err != nil {
		return decoded, errors.New("invalid json")
	}

	return decoded, validatePatchV1TeamRequestBody(decoded)
}

func validatePatchV1TeamRequestBody(body patchV1TeamRequestBody) error {
	if body.DisplayName != nil && len(*body.DisplayName) > name.MaxDisplayLength {
		return fmt.Errorf("display name over character limit: %d", name.MaxDisplayLength)
	}

	if body.DescriptionMarkdown != nil && len(*body.DescriptionMarkdown) > description.MaxLength {
		return fmt.Errorf("description markdown over character limit: %d", description.MaxLength)
	}

	if body.LogoID.Present {
		if body.LogoID.Value != nil && len(*body.LogoID.Value) > image.MaxIDLength {
			return errors.New("invalid logo id")
		}
	}

	if body.BannerID.Present {
		if body.BannerID.Value != nil && len(*body.BannerID.Value) > image.MaxIDLength {
			return errors.New("invalid banner id")
		}
	}

	if body.BackgroundImageID.Present {
		if body.BackgroundImageID.Value != nil && len(*body.BackgroundImageID.Value) > image.MaxIDLength {
			return errors.New("invalid background image id")
		}
	}

	return nil
}

func shouldChangeImage(original *db.Image, newImageID v1.NullString) bool {
	if newImageID.Value == nil {
		return false
	}

	return original == nil || original.ID != *newImageID.Value
}

func shouldRemoveImage(newImageID v1.NullString) bool {
	return newImageID.Present && newImageID.Value == nil
}

// Attempt to find an image from S3 and save it, returning:
// 1. Found image (s3.Image)
// 2. Whether the image was found (bool)
// 3. Any unexpected S3 errors (error)
func (s *Server) findAndSaveImage(ctx context.Context, team db.Team, id, category string) (s3.Image, bool, error) {
	var image s3.Image
	var err error

	image, err = s.s3.Find(ctx, s3.FindParams{
		TeamName: team.Name,
		Category: category,
		ID:       id,
	})

	if err != nil {
		switch err {
		case s3.ErrNoImage, s3.ErrMalformedImage:
			return image, false, nil
		default:
			return image, false, err
		}
	}

	err = s.s3.Save(ctx, s3.SaveParams{
		TeamName: team.Name,
		Category: category,
		ID:       id,
		Format:   image.Format,
	})

	if err != nil {
		switch err {
		case s3.ErrNoImage:
			return image, false, nil
		default:
			return image, false, err
		}
	}

	return image, true, nil
}
