package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"

	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PatchV1TeamMembership", func() {
	var (
		cache    *mocks.Cache
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		server   *Server
		recorder *httptest.ResponseRecorder
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		cache = &mocks.Cache{}
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			Cache:            cache,
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	It("fails with Bad Request with negative display position", func() {
		path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", "123", "456")
		payload := `{"display_position":-1}`
		req, err := http.NewRequest(http.MethodPatch, path, strings.NewReader(payload))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	It("fails with Bad Request with non numeric display position", func() {
		path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", "123", "456")
		payload := `{"display_position":abc}`
		req, err := http.NewRequest(http.MethodPatch, path, strings.NewReader(payload))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	It("fails with Bad Request with empty display position", func() {
		path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", "123", "456")
		payload := `{"display_position": ""}`
		req, err := http.NewRequest(http.MethodPatch, path, strings.NewReader(payload))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))

	})

	It("fails with Bad Request with missing display position", func() {
		path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", "123", "456")
		payload := `{}`
		req, err := http.NewRequest(http.MethodPatch, path, strings.NewReader(payload))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	It("fails with Bad Request with invalid json body", func() {
		path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", "123", "456")
		payload := `{} this is not json`
		req, err := http.NewRequest(http.MethodPatch, path, strings.NewReader(payload))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	Context("when the params are valid", func() {
		var (
			teamID, channelID string
			req               *http.Request
			err               error
		)

		BeforeEach(func() {
			teamID = "123"
			channelID = "456"
			payload := `{"display_position": 2}`

			path := fmt.Sprintf("/v1/teams/%s/channels/%s/membership", teamID, channelID)
			req, err = http.NewRequest(http.MethodPatch, path, strings.NewReader(payload))
			Expect(err).NotTo(HaveOccurred())
		})

		It("fails with Not Found when the team does not exist", func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
		})

		It("fails with Internal Server Error when db errors", func() {
			dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("some error"))

			server.ServeHTTP(recorder, req)

			dbReader.AssertExpectations(GinkgoT())
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
		})

		Context("when the team is found", func() {
			var foundTeam db.Team
			var newDisplayPosition uint

			BeforeEach(func() {
				foundTeam = db.Team{
					ID:     teamID,
					Name:   "some-team-name",
					UserID: "team-owner-id",
				}

				newDisplayPosition = 2

				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(foundTeam, nil)
			})

			It("fails with Not Found when the channel membership is not found", func() {
				dbWriter.On("UpdateMembershipDisplayPosition", mock.Anything, teamID, channelID, newDisplayPosition).
					Return(db.ErrNoMembershipForUpdate)

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				dbWriter.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
			})

			It("fails with Internal Server Error when DB errors", func() {
				dbWriter.On("UpdateMembershipDisplayPosition",
					mock.Anything,
					teamID,
					channelID,
					newDisplayPosition,
				).Return(errors.New("some error"))

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				dbWriter.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			})

			It("succeeds with No Content when the update on db succeeds", func() {
				dbWriter.On("UpdateMembershipDisplayPosition", mock.Anything, teamID, channelID, newDisplayPosition).
					Return(nil)

				cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)

				server.ServeHTTP(recorder, req)

				dbReader.AssertExpectations(GinkgoT())
				dbWriter.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
			})
		})
	})
})
