package api

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/description"
	"code.justin.tv/cb/roster/internal/image"
	"code.justin.tv/cb/roster/internal/name"
	"code.justin.tv/cb/roster/internal/s3"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PatchV1Team", func() {
	var (
		cache    *mocks.Cache
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		mockedS3 *mocks.S3
		users    *mocks.Users
		server   *Server
		recorder *httptest.ResponseRecorder

		teamID, teamName                         string
		team                                     db.Team
		displayName, userID, descriptionMarkdown string
		logoID, bannerID, backgroundID           string
		format                                   string
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		cache = &mocks.Cache{}
		mockedS3 = &mocks.S3{}
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			Cache:            cache,
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			S3:               mockedS3,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		teamID = "123"
		teamName = "NAME"
		displayName = "DISPLAY_NAME"
		userID = "999999999"
		descriptionMarkdown = "[![image](https://www.twitch.tv/kappa.jpg)](https://www.twitch.tv/kappa)"
	})

	AfterEach(func() {
		mockedS3.AssertExpectations(GinkgoT())
		dbWriter.AssertExpectations(GinkgoT())
		users.AssertExpectations(GinkgoT())
	})

	It("fails with Bad Request when team ID is invalid", func() {
		path := fmt.Sprintf("/v1/teams/%s", "=D")

		req, err := http.NewRequest(http.MethodPatch, path, nil)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
		Expect(recorder.Body.String()).To(ContainSubstring("invalid team id (must be numeric)"))
	})

	It("fails with Bad Request when the request body is malformed", func() {
		path := fmt.Sprintf("/v1/teams/%s", teamID)

		req, err := http.NewRequest(http.MethodPatch, path, bytes.NewBufferString("not valid json"))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
		Expect(recorder.Body.String()).To(ContainSubstring("invalid request body: invalid json"))
	})

	It("fails with Bad Request when 'display_name' is over character limit", func() {
		overLengthDisplayName := strings.Repeat("*", name.MaxDisplayLength+1)
		path := fmt.Sprintf("/v1/teams/%s", teamID)

		reqBody := fmt.Sprintf(`{
			"display_name": "%s"
		}`, overLengthDisplayName)

		req, err := http.NewRequest(http.MethodPatch, path, bytes.NewBufferString(reqBody))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
		Expect(recorder.Body.String()).To(ContainSubstring("display name over character limit"))
	})

	It("fails with Bad Request when 'description_markdown' is over character limit", func() {
		overLengthDescriptionMarkdown := strings.Repeat("*", description.MaxLength+1)
		path := fmt.Sprintf("/v1/teams/%s", teamID)

		reqBody := fmt.Sprintf(`{
			"description_markdown": "%s"
		}`, overLengthDescriptionMarkdown)

		req, err := http.NewRequest(http.MethodPatch, path, bytes.NewBufferString(reqBody))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
		Expect(recorder.Body.String()).To(ContainSubstring("description markdown over character limit"))
	})

	Context("when an image id is over-length", func() {
		overLengthID := strings.Repeat("*", image.MaxIDLength+1)

		It("fails with Bad Request when 'logo_id' is over-length", func() {
			path := fmt.Sprintf("/v1/teams/%s", teamID)

			reqBody := fmt.Sprintf(`{
				"logo_id": "%s"
			}`, overLengthID)

			req, err := http.NewRequest(http.MethodPatch, path, bytes.NewBufferString(reqBody))
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid logo id"))
		})

		It("fails with Bad Request when 'banner_id' is over-length", func() {
			path := fmt.Sprintf("/v1/teams/%s", teamID)

			reqBody := fmt.Sprintf(`{
				"banner_id": "%s"
			}`, overLengthID)

			req, err := http.NewRequest(http.MethodPatch, path, bytes.NewBufferString(reqBody))
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid banner id"))
		})

		It("fails with Bad Request when 'background_image_id' is over-length", func() {
			path := fmt.Sprintf("/v1/teams/%s", teamID)

			reqBody := fmt.Sprintf(`{
				"background_image_id": "%s"
			}`, overLengthID)

			req, err := http.NewRequest(http.MethodPatch, path, bytes.NewBufferString(reqBody))
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid background image id"))
		})
	})

	Context("when the request parameters are valid", func() {
		logoID = "LOGO"
		bannerID = "BANNER"
		backgroundID = "BG"
		format = "FORMAT"

		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s", teamID)

			reqBody := v1.PatchTeamRequestBody{
				DisplayName:         &displayName,
				UserID:              &userID,
				DescriptionMarkdown: &descriptionMarkdown,
				LogoID: &v1.NullString{
					Present: true,
					Value:   &logoID,
				},
				BannerID: &v1.NullString{
					Present: true,
					Value:   &bannerID,
				},
				BackgroundImageID: &v1.NullString{
					Present: true,
					Value:   &backgroundID,
				},
			}

			buffer := &bytes.Buffer{}
			err := json.NewEncoder(buffer).Encode(&reqBody)
			Expect(err).NotTo(HaveOccurred())

			req, err := http.NewRequest(http.MethodPatch, path, buffer)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("when db does not find the given team", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
			})

			It("returns Not Found", func() {
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
				Expect(recorder.Body.String()).To(ContainSubstring("not found"))
			})
		})

		Context("when db fails to find the given team", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("db error"))
			})

			It("returns Internal Server Error", func() {
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
				Expect(recorder.Body.String()).To(ContainSubstring("db: failed to find team"))
			})
		})

		Context("when db finds the given team without needing changes to images", func() {
			BeforeEach(func() {
				team = db.Team{
					ID:          teamID,
					Name:        teamName,
					DisplayName: displayName,
					Logo: &db.Image{
						ID: logoID,
					},
					Banner: &db.Image{
						ID: bannerID,
					},
					Background: &db.Image{
						ID: backgroundID,
					},
				}

				team.SetDescriptionWithMarkdown(descriptionMarkdown)

				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(team, nil)
			})

			Context("when the Users service cannot find a user for the given user ID", func() {
				BeforeEach(func() {
					users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
						Return(nil, &channels.ErrChannelNotFound{})
				})

				It("returns Unprocessable Entity", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
					Expect(recorder.Body.String()).To(ContainSubstring("users service:"))
				})
			})

			Context("when the Users service fails", func() {
				BeforeEach(func() {
					users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
						Return(nil, errors.New("💩"))
				})

				It("returns Internal Server Error", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					Expect(recorder.Body.String()).To(ContainSubstring("users service: failed to look up channel"))
				})
			})

			Context("when the Users service succeeds in finding a user for the given user ID", func() {
				BeforeEach(func() {
					users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
						Return(&channels.Channel{}, nil)

					team.UserID = userID
				})

				Context("when db has no team to update", func() {
					BeforeEach(func() {
						dbWriter.On("UpdateTeam", mock.Anything, team).Return(db.ErrNoTeamForUpdate)
					})

					It("returns Not Found", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
						Expect(recorder.Body.String()).To(ContainSubstring("not found"))
					})
				})

				Context("when db fails to update", func() {
					BeforeEach(func() {
						dbWriter.On("UpdateTeam", mock.Anything, team).Return(errors.New("db error"))
					})

					It("returns Internal Server Error", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					})
				})

				Context("when db succeeds to update", func() {
					BeforeEach(func() {
						dbWriter.On("UpdateTeam", mock.Anything, team).Return(nil)
						cache.On("ClearTeam", mock.Anything, teamID).Return(nil)
						cache.On("ClearAllTeams", mock.Anything).Return(nil)

						dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).Return([]db.Membership{
							{ChannelID: "1"},
							{ChannelID: "2"},
						}, nil)

						cache.On("ClearChannelMemberships", mock.Anything, "1").Return(nil)
						cache.On("ClearChannelMemberships", mock.Anything, "2").Return(nil)
					})

					It("returns OK", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusOK))
						Expect(recorder.Body.String()).To(ContainSubstring(logoID))
						Expect(recorder.Body.String()).To(ContainSubstring(bannerID))
						Expect(recorder.Body.String()).To(ContainSubstring(backgroundID))
					})
				})
			})
		})

		Context("when db finds the given team needing changes to images", func() {
			BeforeEach(func() {
				team = db.Team{
					ID:         teamID,
					Name:       teamName,
					Logo:       nil,
					Banner:     nil,
					Background: nil,
				}

				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(team, nil)
			})

			Context("when the Users service cannot find a user for the given user ID", func() {
				BeforeEach(func() {
					users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
						Return(nil, &channels.ErrChannelNotFound{})
				})

				It("returns Unprocessable Entity", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
					Expect(recorder.Body.String()).To(ContainSubstring("users service:"))
				})
			})

			Context("when the Users service fails", func() {
				BeforeEach(func() {
					users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
						Return(nil, errors.New("💩"))
				})

				It("returns Internal Server Error", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					Expect(recorder.Body.String()).To(ContainSubstring("users service: failed to look up channel"))
				})
			})

			Context("when the Users service succeeds in finding a user for the given user ID", func() {
				BeforeEach(func() {
					users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
						Return(&channels.Channel{}, nil)

					team.UserID = userID
				})

				Context("when s3 does not find the logo", func() {
					BeforeEach(func() {
						mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, s3.ErrNoImage).Once()
					})

					It("returns Unprocessable Entity", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
						Expect(recorder.Body.String()).To(ContainSubstring("LOGO not found"))
					})
				})

				Context("when s3 finds malformed logo", func() {
					BeforeEach(func() {
						mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, s3.ErrMalformedImage).Once()
					})

					It("returns Unprocessable Entity", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
						Expect(recorder.Body.String()).To(ContainSubstring("LOGO not found"))
					})
				})

				Context("when s3 fails when attempting to find the logo", func() {
					BeforeEach(func() {
						mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, errors.New("s3 error")).Once()
					})

					It("returns Internal Server Error", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
						Expect(recorder.Body.String()).To(ContainSubstring("failed to find logo"))
					})
				})

				Context("when s3 succeeds in finding the logo", func() {
					BeforeEach(func() {
						mockedS3.On("Find", mock.Anything, s3.FindParams{
							TeamName: teamName,
							Category: image.CategoryLogo,
							ID:       logoID,
						}).Return(s3.Image{
							ID:     logoID,
							Format: format,
						}, nil).Once()
					})

					Context("when s3 saves no logo", func() {
						BeforeEach(func() {
							mockedS3.On("Save", mock.Anything, mock.Anything).Return(s3.ErrNoImage).Once()
						})

						It("returns Unprocessable Entity", func() {
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
							Expect(recorder.Body.String()).To(ContainSubstring("LOGO not found"))
						})
					})

					Context("when s3 fails to save the logo", func() {
						BeforeEach(func() {
							mockedS3.On("Save", mock.Anything, mock.Anything).Return(errors.New("s3 error")).Once()
						})

						It("returns Internal Server Error", func() {
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
							Expect(recorder.Body.String()).To(ContainSubstring("failed to find logo"))
						})
					})

					Context("when s3 succeeds in saving the logo", func() {
						BeforeEach(func() {
							mockedS3.On("Save", mock.Anything, s3.SaveParams{
								TeamName: teamName,
								Category: image.CategoryLogo,
								ID:       logoID,
								Format:   format,
							}).Return(nil).Once()
						})

						Context("when s3 does not find the banner", func() {
							BeforeEach(func() {
								mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, s3.ErrNoImage).Once()
							})

							It("returns Unprocessable Entity", func() {
								Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
								Expect(recorder.Body.String()).To(ContainSubstring("BANNER not found"))
							})
						})

						Context("when s3 finds malformed banner", func() {
							BeforeEach(func() {
								mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, s3.ErrMalformedImage).Once()
							})

							It("returns Unprocessable Entity", func() {
								Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
								Expect(recorder.Body.String()).To(ContainSubstring("BANNER not found"))
							})
						})

						Context("when s3 fails when attempting to find the banner", func() {
							BeforeEach(func() {
								mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, errors.New("s3 error")).Once()
							})

							It("returns Internal Server Error", func() {
								Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
								Expect(recorder.Body.String()).To(ContainSubstring("failed to find banner"))
							})
						})

						Context("when s3 succeeds in finding the banner", func() {
							BeforeEach(func() {
								mockedS3.On("Find", mock.Anything, s3.FindParams{
									TeamName: teamName,
									Category: image.CategoryBanner,
									ID:       bannerID,
								}).Return(s3.Image{
									ID:     bannerID,
									Format: format,
								}, nil).Once()
							})

							Context("when s3 saves no banner", func() {
								BeforeEach(func() {
									mockedS3.On("Save", mock.Anything, mock.Anything).Return(s3.ErrNoImage).Once()
								})

								It("returns Unprocessable Entity", func() {
									Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
									Expect(recorder.Body.String()).To(ContainSubstring("BANNER not found"))
								})
							})

							Context("when s3 fails to save the banner", func() {
								BeforeEach(func() {
									mockedS3.On("Save", mock.Anything, mock.Anything).Return(errors.New("s3 error")).Once()
								})

								It("returns Internal Server Error", func() {
									Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
									Expect(recorder.Body.String()).To(ContainSubstring("failed to find banner"))
								})
							})

							Context("when s3 succeeds in saving the banner", func() {
								BeforeEach(func() {
									mockedS3.On("Save", mock.Anything, s3.SaveParams{
										TeamName: teamName,
										Category: image.CategoryBanner,
										ID:       bannerID,
										Format:   format,
									}).Return(nil).Once()
								})

								Context("when s3 does not find the background image", func() {
									BeforeEach(func() {
										mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, s3.ErrNoImage).Once()
									})

									It("returns Unprocessable Entity", func() {
										Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
										Expect(recorder.Body.String()).To(ContainSubstring("BG not found"))
									})
								})

								Context("when s3 finds malformed background image", func() {
									BeforeEach(func() {
										mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, s3.ErrMalformedImage).Once()
									})

									It("returns Unprocessable Entity", func() {
										Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
										Expect(recorder.Body.String()).To(ContainSubstring("BG not found"))
									})
								})

								Context("when s3 fails when attempting to find the background image", func() {
									BeforeEach(func() {
										mockedS3.On("Find", mock.Anything, mock.Anything).Return(s3.Image{}, errors.New("s3 error")).Once()
									})

									It("returns Internal Server Error", func() {
										Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
										Expect(recorder.Body.String()).To(ContainSubstring("failed to find background image"))
									})
								})

								Context("when s3 succeeds in finding the background image", func() {
									BeforeEach(func() {
										mockedS3.On("Find", mock.Anything, s3.FindParams{
											TeamName: teamName,
											Category: image.CategoryBackground,
											ID:       backgroundID,
										}).Return(s3.Image{
											ID:     backgroundID,
											Format: format,
										}, nil).Once()
									})

									Context("when s3 saves no background image", func() {
										BeforeEach(func() {
											mockedS3.On("Save", mock.Anything, mock.Anything).Return(s3.ErrNoImage).Once()
										})

										It("returns Unprocessable Entity", func() {
											Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
											Expect(recorder.Body.String()).To(ContainSubstring("BG not found"))
										})
									})

									Context("when s3 fails to save the background image", func() {
										BeforeEach(func() {
											mockedS3.On("Save", mock.Anything, mock.Anything).Return(errors.New("s3 error")).Once()
										})

										It("returns Internal Server Error", func() {
											Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
											Expect(recorder.Body.String()).To(ContainSubstring("failed to find background image"))
										})
									})

									Context("when s3 succeeds in saving the background image", func() {
										BeforeEach(func() {
											mockedS3.On("Save", mock.Anything, s3.SaveParams{
												TeamName: teamName,
												Category: image.CategoryBackground,
												ID:       backgroundID,
												Format:   format,
											}).Return(nil).Once()
										})

										Context("when DB succeeds in updating with the metadata of the new images", func() {
											expectedLogoURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-NAME-team_logo_image-LOGO-600x600.FORMAT"
											expectedBannerURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-NAME-banner_image-BANNER-640x125.FORMAT"
											expectedBackgroundURL := "https://static-cdn.jtvnw.net/jtv_user_pictures/team-NAME-background_image-BG.FORMAT"

											BeforeEach(func() {
												team.UserID = userID
												team.DisplayName = displayName
												team.Logo = &db.Image{
													ID:     logoID,
													Format: format,
													URL:    expectedLogoURL,
												}
												team.Banner = &db.Image{
													ID:     bannerID,
													Format: format,
													URL:    expectedBannerURL,
												}
												team.Background = &db.Image{
													ID:     backgroundID,
													Format: format,
													URL:    expectedBackgroundURL,
												}
												team.SetDescriptionWithMarkdown(descriptionMarkdown)

												dbWriter.On("UpdateTeam", mock.Anything, team).Return(nil)
												cache.On("ClearTeam", mock.Anything, teamID).Return(nil)
												cache.On("ClearAllTeams", mock.Anything).Return(nil)

												dbReader.On("GetTeamMemberships", mock.Anything, teamID, mock.Anything).Return([]db.Membership{
													{ChannelID: "1"},
													{ChannelID: "2"},
												}, nil)

												cache.On("ClearChannelMemberships", mock.Anything, "1").Return(nil)
												cache.On("ClearChannelMemberships", mock.Anything, "2").Return(nil)
											})

											It("returns OK", func() {
												dbWriter.AssertExpectations(GinkgoT())
												Expect(recorder.Result().StatusCode).To(Equal(http.StatusOK))
												Expect(recorder.Body.String()).To(ContainSubstring(expectedLogoURL))
												Expect(recorder.Body.String()).To(ContainSubstring(expectedBannerURL))
												Expect(recorder.Body.String()).To(ContainSubstring(expectedBackgroundURL))
											})
										})
									})
								})
							})
						})
					})
				})
			})
		})
	})
})
