package api

import (
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/client/channels"
)

// POST /v1/teams/:team_id/featured_channels
func (s *Server) postV1FeaturedChannels(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()

	teamID := ctx.Value(contextKeyTeamID).(string)

	var reqBody v1.PostFeaturedChannelsRequestBody

	err := json.NewDecoder(req.Body).Decode(&reqBody)
	if err != nil {
		jsonWriter.BadRequest(fmt.Sprint("invalid request body:", err))
		return
	}

	if reqBody.ChannelID == "" {
		jsonWriter.BadRequest("channel id is required")
		return
	}

	_, err = s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to query team", err)
		}
		return
	}

	_, err = s.users.Get(ctx, reqBody.ChannelID, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.NotFound(err.Error())
		default:
			jsonWriter.InternalServerError("users_service: failed to look up channel", err)
		}
		return
	}

	_, err = s.dbReader.GetFeaturedChannel(ctx, teamID, reqBody.ChannelID)
	switch {
	case err == nil:
		jsonWriter.UnprocessableEntity("this channel is already featured by this team")
		return
	case err != db.ErrNoFeaturedChannel:
		jsonWriter.InternalServerError("db: failed to query featured channels", err)
		return
	}

	err = s.dbWriter.CreateFeaturedChannel(ctx, teamID, reqBody.ChannelID)
	if err != nil {
		switch err {
		case db.ErrNoFeaturedChannelCreated:
			jsonWriter.Conflict("failed to create a featured channel")
		default:
			jsonWriter.InternalServerError("db: error in creating a featured channel", err)
		}
		return
	}

	jsonWriter.Created(v1.PostTeamInvitationsResponse{
		Data: v1.PostTeamInvitationsData{
			ChannelID: reqBody.ChannelID,
			TeamID:    teamID,
		},
	})
}
