package api

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PostV1FeaturedChannels", func() {
	var (
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		users    *mocks.Users
		server   *Server
		recorder *httptest.ResponseRecorder
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	It("fails with Bad Request when request body is invalid", func() {
		path := fmt.Sprintf("/v1/teams/%s/featured_channels", "123")
		req, err := http.NewRequest(http.MethodPost, path, bytes.NewReader([]byte{}))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	It("fails with Bad Request when channel ID is invalid", func() {
		path := fmt.Sprintf("/v1/teams/%s/featured_channels", "123")
		reqBody := v1.PostFeaturedChannelsRequestBody{
			ChannelID: "",
		}

		buffer := &bytes.Buffer{}
		err := json.NewEncoder(buffer).Encode(&reqBody)
		Expect(err).NotTo(HaveOccurred())

		req, err := http.NewRequest(http.MethodPost, path, buffer)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	Context("when the request parameters are valid", func() {
		var teamID = "123"
		var channelID = "456"

		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/featured_channels", teamID)
			reqBody := v1.PostFeaturedChannelsRequestBody{
				ChannelID: channelID,
			}

			buffer := new(bytes.Buffer)
			err := json.NewEncoder(buffer).Encode(&reqBody)
			Expect(err).NotTo(HaveOccurred())

			req, err := http.NewRequest(http.MethodPost, path, buffer)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("when the team does not exist", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
			})

			It("fails with Not Found", func() {
				dbReader.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
			})
		})

		Context("when the DB errors while querying for the team", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("Database error"))
			})

			It("fails with Internal Server Error", func() {
				dbReader.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("when the team is found", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
					ID: teamID,
				}, nil)
			})

			AfterEach(func() {
				dbReader.AssertExpectations(GinkgoT())
			})

			Context("when the channel is not found in Users Service", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, &channels.ErrChannelNotFound{})
				})

				It("fails with Not Found", func() {
					users.AssertExpectations(GinkgoT())
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
				})
			})

			Context("when request to the Users Service fails", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).
						Return(&channels.Channel{}, errors.New("some error"))
				})

				It("fails with Internal Server Error", func() {
					users.AssertExpectations(GinkgoT())
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("when the Users Service returns the channel", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(&channels.Channel{
						ID: 456,
					}, nil)
				})

				Context("when the channel is already featured by this team in DB", func() {
					BeforeEach(func() {
						dbReader.On("GetFeaturedChannel", mock.Anything, teamID, channelID).
							Return(db.FeaturedChannel{}, nil)
					})

					It("fails with Unprocessable Entity", func() {
						users.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
					})
				})

				Context("when DB errors while querying for featuring by this team", func() {
					BeforeEach(func() {
						dbReader.On("GetFeaturedChannel", mock.Anything, teamID, channelID).
							Return(db.FeaturedChannel{}, errors.New("some error"))
					})

					It("fails with Internal Server Error", func() {
						dbReader.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					})
				})

				Context("when DB returns no featuring of this channel by this team", func() {
					BeforeEach(func() {
						dbReader.On("GetFeaturedChannel", mock.Anything, teamID, channelID).
							Return(db.FeaturedChannel{}, db.ErrNoFeaturedChannel)
					})

					Context("when the DB insertion results in no rows affected", func() {
						BeforeEach(func() {
							dbWriter.On("CreateFeaturedChannel", mock.Anything, teamID, channelID).
								Return(db.ErrNoFeaturedChannelCreated)
						})

						It("fails with Conflict", func() {
							users.AssertExpectations(GinkgoT())
							dbWriter.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusConflict))
						})
					})

					Context("when DB errors while inserting a featured channel", func() {
						BeforeEach(func() {
							dbWriter.On("CreateFeaturedChannel", mock.Anything, teamID, channelID).
								Return(errors.New("some error"))
						})

						It("fails with Internal Server Error", func() {
							users.AssertExpectations(GinkgoT())
							dbWriter.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
						})
					})

					Context("when DB successfully creates the featured channel", func() {
						BeforeEach(func() {
							dbWriter.On("CreateFeaturedChannel", mock.Anything, teamID, channelID).Return(nil)
						})

						It("succeeds with Created when the featured channel is successfully created", func() {
							users.AssertExpectations(GinkgoT())
							dbWriter.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusCreated))
						})
					})
				})
			})
		})
	})
})
