package api

import (
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/s3/uploader"
)

// POST /v1/teams/:team_id/image_uploads
func (s *Server) postV1ImageUploads(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)
	reqBody := v1.PostImageUploadsRequestBody{}

	err := json.NewDecoder(req.Body).Decode(&reqBody)
	if err != nil {
		jsonWriter.BadRequest("invalid request body")
		return
	}

	team, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err == db.ErrNoTeam {
		jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		return
	}
	if err != nil {
		jsonWriter.InternalServerError("db: failed to query team", err)
		return
	}

	upload, err := s.s3Uploader.NewImageUpload(uploader.ImageUploadParams{
		Category:      reqBody.Category,
		ContentLength: reqBody.Size,
		ContentType:   reqBody.FileType,
		TeamName:      team.Name,
	})

	if err != nil {
		switch err {
		case uploader.ErrMissingContentLength:
			jsonWriter.BadRequest("missing image size")
		case uploader.ErrContentLengthOverLimit:
			jsonWriter.BadRequest("image size over limit")
		case uploader.ErrInvalidCategory:
			jsonWriter.BadRequest("invalid image category")
		case uploader.ErrInvalidContentType:
			jsonWriter.BadRequest("invalid image file type")
		default:
			jsonWriter.InternalServerError("failed to create image upload url", err)
		}
		return
	}

	payload := v1.PostImageUploadsResponse{
		Data: v1.PostImageUploadsData{
			ImageID: upload.ImageID,
			URL:     upload.URL,
		},
	}

	jsonWriter.OK(payload)
}
