package api

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/s3/uploader"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PostV1ImageUploads", func() {
	var (
		dbReader *mocks.DBReader
		mockedS3 *mocks.S3Uploader
		server   *Server
		recorder *httptest.ResponseRecorder
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbReader = &mocks.DBReader{}
		mockedS3 = &mocks.S3Uploader{}

		server = NewServer(&ServerParams{
			S3Uploader:       mockedS3,
			DBReader:         dbReader,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	It("fails with Bad Request when request body is invalid", func() {
		path := fmt.Sprintf("/v1/teams/%s/image_uploads", "123")
		req, err := http.NewRequest(http.MethodPost, path, bytes.NewReader([]byte{}))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	Context("when the request parameters are valid", func() {
		teamID := "123"
		teamName := "some team"
		category := "background"
		size := uint64(123)
		fileType := "png"

		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/image_uploads", teamID)
			reqBody := v1.PostImageUploadsRequestBody{
				Category: category,
				Size:     size,
				FileType: fileType,
			}

			buffer := new(bytes.Buffer)
			err := json.NewEncoder(buffer).Encode(&reqBody)
			Expect(err).NotTo(HaveOccurred())

			req, err := http.NewRequest(http.MethodPost, path, buffer)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("when the team does not exist", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
			})

			It("fails with Not Found", func() {
				dbReader.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
			})
		})

		Context("when the DB errors while querying for the team", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("Database error"))
			})

			It("fails with Internal Server Error", func() {
				dbReader.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("when the team is found", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
					ID:   teamID,
					Name: teamName,
				}, nil)
			})

			AfterEach(func() {
				dbReader.AssertExpectations(GinkgoT())
			})

			Context("when category is invalid", func() {
				BeforeEach(func() {
					mockedS3.On("NewImageUpload", mock.Anything).
						Return(uploader.ImageUpload{}, uploader.ErrInvalidCategory)
				})

				It("fails with Bad Request", func() {
					mockedS3.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
				})
			})

			Context("when content length is missing", func() {
				BeforeEach(func() {
					mockedS3.On("NewImageUpload", mock.Anything).
						Return(uploader.ImageUpload{}, uploader.ErrMissingContentLength)
				})

				It("fails with Bad Request", func() {
					mockedS3.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
				})
			})

			Context("when content length is over limit", func() {
				BeforeEach(func() {
					mockedS3.On("NewImageUpload", mock.Anything).
						Return(uploader.ImageUpload{}, uploader.ErrContentLengthOverLimit)
				})

				It("fails with Bad Request", func() {
					mockedS3.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
				})
			})

			Context("when file type is invalid", func() {
				BeforeEach(func() {
					mockedS3.On("NewImageUpload", mock.Anything).
						Return(uploader.ImageUpload{}, uploader.ErrInvalidContentType)
				})

				It("fails with Bad Request", func() {
					mockedS3.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
				})
			})

			Context("when team name is invalid", func() {
				BeforeEach(func() {
					mockedS3.On("NewImageUpload", mock.Anything).
						Return(uploader.ImageUpload{}, uploader.ErrMissingTeamName)
				})

				It("fails with Internal Server Error", func() {
					mockedS3.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("when s3 image upload parameters are valid", func() {
				imageID := "some image id"
				url := "some url"

				BeforeEach(func() {
					mockedS3.On("NewImageUpload", mock.Anything).Return(uploader.ImageUpload{
						ImageID: imageID,
						URL:     url,
					}, nil)
				})

				It("returns the image id and url", func() {
					mockedS3.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusOK))

					response := v1.PostImageUploadsResponse{}
					err := json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(response.Data.ImageID).To(Equal(imageID))
					Expect(response.Data.URL).To(Equal(url))
				})
			})
		})
	})
})
