package api

import (
	"context"
	"encoding/json"
	"fmt"
	"net/http"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/client/channels"
	log "github.com/sirupsen/logrus"
)

// POST /v1/teams/:team_id/invitations
func (s *Server) postV1TeamInvitations(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	reqBody := v1.PostTeamInvitationsRequestBody{}

	err := json.NewDecoder(req.Body).Decode(&reqBody)
	if err != nil {
		jsonWriter.BadRequest(fmt.Sprint("invalid request body:", err))
		return
	}

	if reqBody.ChannelID == "" {
		jsonWriter.BadRequest("channel id is required")
		return
	}

	team, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to query team", err)
		}
		return
	}

	_, err = s.users.Get(ctx, reqBody.ChannelID, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.NotFound(err.Error())
		default:
			jsonWriter.InternalServerError("users_service: failed to look up channel", err)
		}
		return
	}

	_, err = s.dbReader.GetMembership(ctx, teamID, reqBody.ChannelID)
	switch {
	case err == nil:
		jsonWriter.UnprocessableEntity("channel with an existing team membership cannot be invited")
		return
	case err != db.ErrNoMembership:
		jsonWriter.InternalServerError("db: failed to query membership", err)
		return
	}

	_, err = s.dbReader.GetInvitation(ctx, teamID, reqBody.ChannelID)
	switch {
	case err == nil:
		jsonWriter.UnprocessableEntity("channel with a pending team invitation cannot be invited again")
		return
	case err != db.ErrNoInvitation:
		jsonWriter.InternalServerError("db: failed to query invitation", err)
		return
	}

	err = s.dbWriter.CreateInvitation(ctx, teamID, reqBody.ChannelID)
	if err != nil {
		switch err {
		case db.ErrNoInvitationCreated:
			jsonWriter.Conflict("failed to create an invitation")
		default:
			jsonWriter.InternalServerError("db: error in creating an invitation", err)
		}
		return
	}

	go func() {
		pushyErr := s.pushy.PublishInvite(context.Background(), team.UserID, team.DisplayName, reqBody.ChannelID)
		if pushyErr != nil {
			log.WithError(pushyErr).WithFields(log.Fields{
				"team_id":    team.ID,
				"channel_id": reqBody.ChannelID,
			}).Error("pushy: failed to publish invitation")
		}
	}()

	jsonWriter.Created(v1.PostTeamInvitationsResponse{
		Data: v1.PostTeamInvitationsData{
			ChannelID: reqBody.ChannelID,
			TeamID:    teamID,
		},
	})
}
