package api

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PostV1TeamInvitations", func() {
	var (
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		pushy    *mocks.Pushy
		users    *mocks.Users
		server   *Server
		recorder *httptest.ResponseRecorder
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}
		pushy = &mocks.Pushy{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			Pushy:            pushy,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	It("fails with Bad Request when request body is invalid", func() {
		path := fmt.Sprintf("/v1/teams/%s/invitations", "123")
		req, err := http.NewRequest(http.MethodPost, path, bytes.NewReader([]byte{}))
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	It("fails with Bad Request when channel ID is invalid", func() {
		path := fmt.Sprintf("/v1/teams/%s/invitations", "123")
		reqBody := v1.PostTeamInvitationsRequestBody{
			ChannelID: "",
		}

		buffer := &bytes.Buffer{}
		err := json.NewEncoder(buffer).Encode(&reqBody)
		Expect(err).NotTo(HaveOccurred())

		req, err := http.NewRequest(http.MethodPost, path, buffer)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
	})

	Context("when the request parameters are valid", func() {
		var teamID = "123"
		var teamUserID = "345"
		var teamName = "TeamName"
		var channelID = "456"

		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/teams/%s/invitations", teamID)
			reqBody := v1.PostTeamInvitationsRequestBody{
				ChannelID: channelID,
			}

			buffer := new(bytes.Buffer)
			err := json.NewEncoder(buffer).Encode(&reqBody)
			Expect(err).NotTo(HaveOccurred())

			req, err := http.NewRequest(http.MethodPost, path, buffer)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("when the team does not exist", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, db.ErrNoTeam)
			})

			It("fails with Not Found", func() {
				dbReader.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
			})
		})

		Context("when the DB errors while querying for the team", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{}, errors.New("Database error"))
			})

			It("fails with Internal Server Error", func() {
				dbReader.AssertExpectations(GinkgoT())
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("when the team is found", func() {
			BeforeEach(func() {
				dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
					ID:          teamID,
					Name:        teamName,
					DisplayName: teamName,
					UserID:      teamUserID,
				}, nil)
			})

			AfterEach(func() {
				dbReader.AssertExpectations(GinkgoT())
			})

			Context("when the channel is not found in Users Service", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, &channels.ErrChannelNotFound{})
				})

				It("fails with Not Found", func() {
					users.AssertExpectations(GinkgoT())
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
				})
			})

			Context("when request to the Users Service fails", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(&channels.Channel{}, errors.New("some error"))
				})

				It("fails with Internal Server Error", func() {
					users.AssertExpectations(GinkgoT())
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
				})
			})

			Context("when the Users Service returns the channel", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(&channels.Channel{
						ID: 456,
					}, nil)
				})

				Context("when the channel is already a member in DB", func() {
					BeforeEach(func() {
						dbReader.On("GetMembership", mock.Anything, teamID, channelID).Return(db.Membership{}, nil)
					})

					It("fails with Unprocessable Entity", func() {
						users.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
					})
				})

				Context("when DB errors while querying for a membership", func() {
					BeforeEach(func() {
						dbReader.On("GetMembership", mock.Anything, teamID, channelID).
							Return(db.Membership{}, errors.New("some error"))
					})

					It("fails with Internal Server Error", func() {
						users.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					})
				})

				Context("when DB returns no matching membership", func() {
					BeforeEach(func() {
						dbReader.On("GetMembership", mock.Anything, teamID, channelID).
							Return(db.Membership{}, db.ErrNoMembership)
					})

					Context("when the channel has an existing invitation", func() {
						BeforeEach(func() {
							dbReader.On("GetInvitation", mock.Anything, teamID, channelID).Return(db.Invitation{}, nil)
						})

						It("fails with Unprocessable Entity", func() {
							users.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
						})
					})

					Context("when DB errors while querying for an invitation", func() {
						BeforeEach(func() {
							dbReader.On("GetInvitation", mock.Anything, teamID, channelID).
								Return(db.Invitation{}, errors.New("some error"))
						})

						It("fails with Internal Server Error", func() {
							users.AssertExpectations(GinkgoT())
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
						})
					})

					Context("when DB returns no matching invitation", func() {
						BeforeEach(func() {
							dbReader.On("GetInvitation", mock.Anything, teamID, channelID).
								Return(db.Invitation{}, db.ErrNoInvitation)
						})

						Context("when the DB insertion results in no rows affected", func() {
							BeforeEach(func() {
								dbWriter.On("CreateInvitation", mock.Anything, teamID, channelID).Return(db.ErrNoInvitationCreated)
							})

							It("fails with Conflict", func() {
								users.AssertExpectations(GinkgoT())
								dbWriter.AssertExpectations(GinkgoT())
								Expect(recorder.Result().StatusCode).To(Equal(http.StatusConflict))
							})
						})

						Context("when DB errors while inserting an invitation", func() {
							BeforeEach(func() {
								dbWriter.On("CreateInvitation", mock.Anything, teamID, channelID).Return(errors.New("some error"))
							})

							It("fails with Internal Server Error", func() {
								users.AssertExpectations(GinkgoT())
								dbWriter.AssertExpectations(GinkgoT())
								Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
							})
						})

						Context("when DB successfully creates the invitation", func() {
							BeforeEach(func() {
								dbWriter.On("CreateInvitation", mock.Anything, teamID, channelID).Return(nil)
								pushy.On("PublishInvite", mock.Anything, teamUserID, teamName, channelID).Return(nil)
							})

							It("succeeds with Created when the invitation is successfully created", func() {
								users.AssertExpectations(GinkgoT())
								dbWriter.AssertExpectations(GinkgoT())

								Expect(recorder.Result().StatusCode).To(Equal(http.StatusCreated))
							})
						})
					})
				})
			})
		})
	})
})
