package api

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"strconv"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

// POST /v1/teams/:team_id/memberships
func (s *Server) postV1TeamMemberships(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)

	teamIDUint64, err := strconv.ParseUint(teamID, 10, 64)
	if err != nil {
		jsonWriter.BadRequest("team ID must be a positive numeric string")
		return
	}

	reqBody, err := decodePostV1TeamMembershipsRequestBody(req.Body)
	if err != nil {
		jsonWriter.BadRequest(fmt.Sprint("invalid request body: ", err))
		return
	}

	_, err = s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with ID %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to query team", err)
		}
		return
	}

	channel, err := s.users.GetByIDAndParams(ctx, reqBody.ChannelID, &models.ChannelFilterParams{
		NotDeleted:      true,
		NoTOSViolation:  true,
		NoDMCAViolation: true,
	}, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.UnprocessableEntity(err.Error())
		default:
			jsonWriter.InternalServerError("users service: failed to look up channel", err)
		}
		return
	}

	_, err = s.dbReader.GetMembership(ctx, teamID, reqBody.ChannelID)
	if err != nil {
		if err != db.ErrNoMembership {
			jsonWriter.InternalServerError("db: failed to query membership", err)
			return
		}
	} else {
		jsonWriter.UnprocessableEntity("channel already has an existing team membership")
		return
	}

	err = s.dbWriter.CreateMembership(ctx, db.Membership{
		ChannelID:       reqBody.ChannelID,
		TeamID:          teamID,
		RevenueRevealed: *reqBody.RevenueRevealed,
		StatsRevealed:   *reqBody.StatsRevealed,
	})
	if err != nil {
		switch err {
		case db.ErrNoMembershipCreated:
			jsonWriter.Conflict("failed to create membership")
		default:
			jsonWriter.InternalServerError("db: failed to insert a membership record", err)
		}
		return
	}

	shouldSetPrimaryTeam := channel.PrimaryTeamID == 0
	if shouldSetPrimaryTeam {
		err = s.users.Set(ctx, models.UpdateChannelProperties{
			ID:            uint64(channel.ID),
			PrimaryTeamID: &teamIDUint64,
		}, nil)
		if err != nil {
			jsonWriter.InternalServerError("users service: failed to update channel's primary team ID", err)
			return
		}
	}

	go s.expireCachedChannelMemberships(context.Background(), reqBody.ChannelID)
	go s.expireCachedTeamMemberships(context.Background(), teamID)

	jsonWriter.Created(v1.PostTeamMembershipsResponse{
		Data: v1.PostTeamMembershipsData{
			ChannelID:       reqBody.ChannelID,
			TeamID:          teamID,
			Primary:         shouldSetPrimaryTeam,
			RevenueRevealed: *reqBody.RevenueRevealed,
			StatsRevealed:   *reqBody.StatsRevealed,
		},
	})
}

func decodePostV1TeamMembershipsRequestBody(reqBody io.ReadCloser) (v1.PostTeamMembershipsRequestBody, error) {
	var body v1.PostTeamMembershipsRequestBody

	if err := json.NewDecoder(reqBody).Decode(&body); err != nil {
		return body, err
	}

	return body, validatePostV1TeamMembershipsRequestBody(body)
}

func validatePostV1TeamMembershipsRequestBody(body v1.PostTeamMembershipsRequestBody) error {
	if body.ChannelID == "" {
		return errors.New("channel ID cannot be empty")
	}

	if body.RevenueRevealed == nil {
		return errors.New("revenue revealed must be true or false")
	}

	if body.StatsRevealed == nil {
		return errors.New("stats revealed must be true or false")
	}

	return nil
}
