package api

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"net/http"
	"strings"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/description"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/cb/roster/internal/name"
	"code.justin.tv/web/users-service/client/channels"
	"code.justin.tv/web/users-service/models"
)

// POST /v1/users/:user_id/teams
func (s *Server) postV1Teams(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	userID := ctx.Value(contextKeyUserID).(string)

	reqBody, err := decodePostV1TeamsRequestBody(req.Body)
	if err != nil {
		jsonWriter.BadRequest(fmt.Sprint("invalid request body: ", err))
		return
	}

	_, err = s.users.GetByIDAndParams(ctx, userID, &models.ChannelFilterParams{
		NotDeleted:      true,
		NoTOSViolation:  true,
		NoDMCAViolation: true,
	}, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.NotFound(err.Error())
		default:
			jsonWriter.InternalServerError("users service: failed to look up channel", err)
		}
		return
	}

	team, err := s.dbWriter.CreateTeam(ctx, db.CreateTeamParams{
		Name:                reqBody.Name,
		DisplayName:         reqBody.DisplayName,
		UserID:              userID,
		DescriptionMarkdown: reqBody.DescriptionMarkdown,
	})
	if err != nil {
		switch err {
		case db.ErrNoTeamCreated:
			jsonWriter.UnprocessableEntity(fmt.Sprint("team already exists with name: ", reqBody.Name))
		default:
			jsonWriter.InternalServerError("db: failed to create team", err)
		}
		return
	}

	go s.expireCachedTeams(context.Background())

	jsonWriter.Created(v1.PostTeamsResponse{
		Data: transformDBTeamToV1Team(team),
	})
}

func decodePostV1TeamsRequestBody(reqBody io.ReadCloser) (v1.PostTeamsRequestBody, error) {
	var body v1.PostTeamsRequestBody

	if err := json.NewDecoder(reqBody).Decode(&body); err != nil {
		return body, err
	}

	return body, validatePostV1TeamsRequestBody(body)
}

func validatePostV1TeamsRequestBody(body v1.PostTeamsRequestBody) error {
	if body.Name == "" {
		return errors.New("name cannot be empty")
	}

	if len(body.Name) < name.MinLength {
		return fmt.Errorf("name under minimum character limit: %d", name.MinLength)
	}

	if len(body.Name) > name.MaxLength {
		return fmt.Errorf("name over maximum character limit: %d", name.MaxLength)
	}

	body.Name = strings.ToLower(body.Name)

	if ok := name.Valid(body.Name); !ok {
		return errors.New("name contains invalid character(s) or character sequence")
	}

	if body.DisplayName == "" {
		return errors.New("display name cannot be empty")
	}

	if len(body.DisplayName) > name.MaxDisplayLength {
		return fmt.Errorf("display name over character limit: %d", name.MaxDisplayLength)
	}

	if body.DescriptionMarkdown != nil && len(*body.DescriptionMarkdown) > description.MaxLength {
		return fmt.Errorf("description markdown over character limit: %d", description.MaxLength)
	}

	return nil
}
