package api

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"
	"strings"
	"time"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/description"
	"code.justin.tv/cb/roster/internal/name"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PostV1Teams", func() {
	var (
		recorder *httptest.ResponseRecorder
		cache    *mocks.Cache
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		users    *mocks.Users
		server   *Server

		userID string
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		cache = &mocks.Cache{}
		dbReader = &mocks.DBReader{}
		dbWriter = &mocks.DBWriter{}
		users = &mocks.Users{}

		server = NewServer(&ServerParams{
			Cache:            cache,
			DBReader:         dbReader,
			DBWriter:         dbWriter,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	Context("when the user ID path param is invalid", func() {
		BeforeEach(func() {
			userID = "non-numeric"

			path := fmt.Sprintf("/v1/users/%s/teams", userID)
			req, err := http.NewRequest(http.MethodPost, path, nil)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		It("fails with Bad Request", func() {
			Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid user id (must be numeric)"))
		})
	})

	Context("when the user ID path param is valid", func() {
		var (
			buffer                *bytes.Buffer
			teamName, displayName string
			descriptionMarkdown   *string
			reqBody               v1.PostTeamsRequestBody
		)

		BeforeEach(func() {
			userID = "123"
		})

		Context("when the request body is invalid", func() {
			Context("when the name is missing", func() {
				BeforeEach(func() {
					reqBody = v1.PostTeamsRequestBody{
						Name:                "",
						DisplayName:         displayName,
						DescriptionMarkdown: descriptionMarkdown,
					}

					buffer = new(bytes.Buffer)
					err := json.NewEncoder(buffer).Encode(&reqBody)
					Expect(err).NotTo(HaveOccurred())

					path := fmt.Sprintf("/v1/users/%s/teams", userID)
					req, err := http.NewRequest(http.MethodPost, path, buffer)
					Expect(err).NotTo(HaveOccurred())

					server.ServeHTTP(recorder, req)
				})

				It("fails with Bad Request", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
					Expect(recorder.Body.String()).To(ContainSubstring("name cannot be empty"))
				})
			})

			Context("when the name is under the minimum character limit", func() {
				BeforeEach(func() {
					reqBody = v1.PostTeamsRequestBody{
						Name:                strings.Repeat("a", name.MinLength-1),
						DisplayName:         displayName,
						DescriptionMarkdown: descriptionMarkdown,
					}

					buffer = new(bytes.Buffer)
					err := json.NewEncoder(buffer).Encode(&reqBody)
					Expect(err).NotTo(HaveOccurred())

					path := fmt.Sprintf("/v1/users/%s/teams", userID)
					req, err := http.NewRequest(http.MethodPost, path, buffer)
					Expect(err).NotTo(HaveOccurred())

					server.ServeHTTP(recorder, req)
				})

				It("fails with Bad Request", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
					Expect(recorder.Body.String()).To(ContainSubstring("name under minimum character limit"))
				})
			})

			Context("when the name is over the maximum character limit", func() {
				BeforeEach(func() {
					reqBody = v1.PostTeamsRequestBody{
						Name:                strings.Repeat("a", name.MaxLength+1),
						DisplayName:         displayName,
						DescriptionMarkdown: descriptionMarkdown,
					}

					buffer = new(bytes.Buffer)
					err := json.NewEncoder(buffer).Encode(&reqBody)
					Expect(err).NotTo(HaveOccurred())

					path := fmt.Sprintf("/v1/users/%s/teams", userID)
					req, err := http.NewRequest(http.MethodPost, path, buffer)
					Expect(err).NotTo(HaveOccurred())

					server.ServeHTTP(recorder, req)
				})

				It("fails with Bad Request", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
					Expect(recorder.Body.String()).To(ContainSubstring("name over maximum character limit"))
				})
			})

			Context("when the name is invalid", func() {
				BeforeEach(func() {
					reqBody = v1.PostTeamsRequestBody{
						Name:                "[]-INVALID_*()",
						DisplayName:         displayName,
						DescriptionMarkdown: descriptionMarkdown,
					}

					buffer = new(bytes.Buffer)
					err := json.NewEncoder(buffer).Encode(&reqBody)
					Expect(err).NotTo(HaveOccurred())

					path := fmt.Sprintf("/v1/users/%s/teams", userID)
					req, err := http.NewRequest(http.MethodPost, path, buffer)
					Expect(err).NotTo(HaveOccurred())

					server.ServeHTTP(recorder, req)
				})

				It("fails with Bad Request", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
					Expect(recorder.Body.String()).To(ContainSubstring("name contains invalid character(s) or character sequence"))
				})
			})

			Context("when the name is valid", func() {
				BeforeEach(func() {
					teamName = "VALID-NAME_YAY"
				})

				Context("when the display name is missing", func() {
					BeforeEach(func() {
						reqBody = v1.PostTeamsRequestBody{
							Name:                teamName,
							DisplayName:         "",
							DescriptionMarkdown: descriptionMarkdown,
						}

						buffer = new(bytes.Buffer)
						err := json.NewEncoder(buffer).Encode(&reqBody)
						Expect(err).NotTo(HaveOccurred())

						path := fmt.Sprintf("/v1/users/%s/teams", userID)
						req, err := http.NewRequest(http.MethodPost, path, buffer)
						Expect(err).NotTo(HaveOccurred())

						server.ServeHTTP(recorder, req)
					})

					It("fails with Bad Request", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
						Expect(recorder.Body.String()).To(ContainSubstring("display name cannot be empty"))
					})
				})

				Context("when the display name is over the character limit", func() {
					BeforeEach(func() {
						reqBody = v1.PostTeamsRequestBody{
							Name:                teamName,
							DisplayName:         strings.Repeat("d", name.MaxDisplayLength+1),
							DescriptionMarkdown: descriptionMarkdown,
						}

						buffer = new(bytes.Buffer)
						err := json.NewEncoder(buffer).Encode(&reqBody)
						Expect(err).NotTo(HaveOccurred())

						path := fmt.Sprintf("/v1/users/%s/teams", userID)
						req, err := http.NewRequest(http.MethodPost, path, buffer)
						Expect(err).NotTo(HaveOccurred())

						server.ServeHTTP(recorder, req)
					})

					It("fails with Bad Request", func() {
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
						Expect(recorder.Body.String()).To(ContainSubstring("display name over character limit"))
					})
				})

				Context("when the display name is valid", func() {
					BeforeEach(func() {
						displayName = "Valid Name Muahaha"
					})

					Context("when the description is over the character limit", func() {
						BeforeEach(func() {
							overLength := strings.Repeat("*", description.MaxLength+1)

							reqBody = v1.PostTeamsRequestBody{
								Name:                teamName,
								DisplayName:         displayName,
								DescriptionMarkdown: &overLength,
							}

							buffer = new(bytes.Buffer)
							err := json.NewEncoder(buffer).Encode(&reqBody)
							Expect(err).NotTo(HaveOccurred())

							path := fmt.Sprintf("/v1/users/%s/teams", userID)
							req, err := http.NewRequest(http.MethodPost, path, buffer)
							Expect(err).NotTo(HaveOccurred())

							server.ServeHTTP(recorder, req)
						})

						It("fails with Bad Request", func() {
							Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
							Expect(recorder.Body.String()).To(ContainSubstring("description markdown over character limit"))
						})
					})
				})
			})
		})
	})

	Context("when all params are valid", func() {
		var (
			buffer                *bytes.Buffer
			teamName, displayName string
			descriptionMarkdown   *string
			reqBody               v1.PostTeamsRequestBody
			req                   *http.Request
			err                   error
		)

		BeforeEach(func() {
			userID = "123"
			teamName = "VALID-NAME_YAY"
			displayName = "Valid Name Muahaha"
			markdown := "# Header"
			descriptionMarkdown = &markdown

			reqBody = v1.PostTeamsRequestBody{
				Name:                teamName,
				DisplayName:         displayName,
				DescriptionMarkdown: descriptionMarkdown,
			}

			buffer = new(bytes.Buffer)
			err = json.NewEncoder(buffer).Encode(&reqBody)
			Expect(err).NotTo(HaveOccurred())

			path := fmt.Sprintf("/v1/users/%s/teams", userID)
			req, err = http.NewRequest(http.MethodPost, path, buffer)
			Expect(err).NotTo(HaveOccurred())
		})

		Context("when the users service returns 404", func() {
			BeforeEach(func() {
				users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
					Return(nil, &channels.ErrChannelNotFound{})

				server.ServeHTTP(recorder, req)
			})

			It("returns Not Found", func() {
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
				users.AssertExpectations(GinkgoT())
			})
		})

		Context("when the users service returns a non-404 error", func() {
			BeforeEach(func() {
				users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
					Return(nil, errors.New("💩"))

				server.ServeHTTP(recorder, req)
			})

			It("returns Internal Server Error", func() {
				Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
				Expect(recorder.Body.String()).To(ContainSubstring("users service: failed to look up channel"))
				users.AssertExpectations(GinkgoT())
			})
		})

		Context("when the users service returns a valid user", func() {
			var createParams db.CreateTeamParams

			BeforeEach(func() {
				createParams = db.CreateTeamParams{
					Name:                teamName,
					DisplayName:         displayName,
					UserID:              userID,
					DescriptionMarkdown: descriptionMarkdown,
				}

				users.On("GetByIDAndParams", mock.Anything, userID, mock.Anything, mock.Anything).
					Return(&channels.Channel{}, nil)
			})

			AfterEach(func() {
				users.AssertExpectations(GinkgoT())
			})

			Context("when DB creates no team", func() {
				BeforeEach(func() {
					dbWriter.On("CreateTeam", mock.Anything, createParams).Return(db.Team{}, db.ErrNoTeamCreated)

					server.ServeHTTP(recorder, req)
				})

				AfterEach(func() {
					dbWriter.AssertExpectations(GinkgoT())
				})

				It("returns Unprocessable Entity", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusUnprocessableEntity))
					Expect(recorder.Body.String()).To(ContainSubstring("team already exists with name"))
				})
			})

			Context("when DB unexpectedly fails to create team", func() {
				BeforeEach(func() {
					dbWriter.On("CreateTeam", mock.Anything, createParams).Return(db.Team{}, errors.New("🚨"))

					server.ServeHTTP(recorder, req)
				})

				AfterEach(func() {
					dbWriter.AssertExpectations(GinkgoT())
				})

				It("returns Internal Server Error", func() {
					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					Expect(recorder.Body.String()).To(ContainSubstring("db: failed to create team"))
				})
			})

			Context("when DB successfully creates the team", func() {
				var (
					id        string
					createdAt time.Time
				)

				BeforeEach(func() {
					id = "1234567890"
					createdAt = time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)

					team := db.Team{
						ID:          id,
						Name:        teamName,
						DisplayName: displayName,
						UserID:      userID,
						CreatedAt:   &createdAt,
					}

					team.SetDescriptionWithMarkdown(*descriptionMarkdown)

					dbWriter.On("CreateTeam", mock.Anything, createParams).Return(team, nil).Once()
					cache.On("ClearAllTeams", mock.Anything).Return(nil)

					server.ServeHTTP(recorder, req)
				})

				AfterEach(func() {
					dbWriter.AssertExpectations(GinkgoT())
				})

				It("returns Created", func() {
					expectedTeam := v1.Team{
						ID:                  id,
						Name:                teamName,
						DisplayName:         displayName,
						UserID:              userID,
						DescriptionHTML:     "<h1>Header</h1>\n",
						DescriptionMarkdown: *descriptionMarkdown,
						CreatedAt:           &createdAt,
					}

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusCreated))

					response := v1.PostTeamsResponse{}
					err := json.Unmarshal(recorder.Body.Bytes(), &response)
					Expect(err).NotTo(HaveOccurred())

					Expect(response.Data).To(Equal(expectedTeam))
				})
			})
		})
	})
})
