package api

import (
	"context"
	"fmt"
	"net/http"

	"encoding/json"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/client/channels"
	log "github.com/sirupsen/logrus"
)

// PUT /v1/channels/:channel_id/teams/:team_id/invitation
func (s *Server) putV1ChannelInvitation(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	teamID := ctx.Value(contextKeyTeamID).(string)
	channelID := ctx.Value(contextKeyChannelID).(string)

	reqBody := v1.PutChannelInvitationRequestBody{}

	err := json.NewDecoder(req.Body).Decode(&reqBody)
	if err != nil {
		jsonWriter.BadRequest(fmt.Sprint("invalid request body:", err))
		return
	}

	if reqBody.Accepted == nil {
		jsonWriter.BadRequest("accept status is required")
		return
	}

	team, err := s.dbReader.GetTeamByID(ctx, teamID)
	if err != nil {
		switch err {
		case db.ErrNoTeam:
			jsonWriter.NotFound(fmt.Sprintf("team with id %s not found", teamID))
		default:
			jsonWriter.InternalServerError("db: failed to query team", err)
		}
		return
	}

	_, err = s.users.Get(ctx, channelID, nil)
	if err != nil {
		switch err.(type) {
		case *channels.ErrChannelNotFound:
			jsonWriter.NotFound(fmt.Sprintf("channel with id %s not found", channelID))
		default:
			jsonWriter.InternalServerError("user service internal error", err)
		}
		return
	}

	if *reqBody.Accepted {
		err = s.dbWriter.AcceptInvitation(ctx, teamID, channelID)
		if err != nil {
			switch err {
			case db.ErrNoRowFoundForDeletion:
				jsonWriter.NotFound("invitation not found")
			case db.ErrNoMembershipCreated:
				jsonWriter.InternalServerError("membership not created successfully", err)
			default:
				jsonWriter.InternalServerError("db internal error", err)
			}
			return
		}

		go s.expireCachedChannelMemberships(context.Background(), channelID)
		go s.expireCachedTeamMemberships(context.Background(), teamID)

		go func() {
			err = s.pushy.PublishInviteAccepted(context.Background(), team.UserID, team.DisplayName, channelID)
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"team_id":    team.ID,
					"channel_id": channelID,
				}).Error("pushy: failed to publish invitation acceptance")
			}
		}()
	} else {
		err = s.dbWriter.DeleteInvitation(ctx, teamID, channelID)
		if err != nil {
			switch err {
			case db.ErrNoRowFoundForDeletion:
				jsonWriter.NotFound("invitation not found")
			default:
				jsonWriter.InternalServerError("db internal error", err)
			}
			return
		}

		go func() {
			err = s.pushy.PublishInviteDeclined(context.Background(), team.UserID, team.DisplayName, channelID)
			if err != nil {
				log.WithError(err).WithFields(log.Fields{
					"team_id":    team.ID,
					"channel_id": channelID,
				}).Error("pushy: failed to publish invitation decline")
			}
		}()
	}

	w.WriteHeader(http.StatusNoContent)
}
