package api

import (
	"bytes"
	"encoding/json"
	"errors"
	"fmt"
	"net/http"
	"net/http/httptest"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/web/users-service/client/channels"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("PutV1ChannelInvitation", func() {
	var (
		cache    *mocks.Cache
		dbWriter *mocks.DBWriter
		dbReader *mocks.DBReader
		pushy    *mocks.Pushy
		users    *mocks.Users

		server   *Server
		recorder *httptest.ResponseRecorder

		teamID     = "123"
		teamName   = "TeamName"
		teamUserID = "333"
		channelID  = "456"
		accepted   = true
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		cache = &mocks.Cache{}
		dbWriter = &mocks.DBWriter{}
		dbReader = &mocks.DBReader{}
		pushy = &mocks.Pushy{}
		users = &mocks.Users{}

		dbReader.On("GetTeamByID", mock.Anything, teamID).Return(db.Team{
			ID: teamID, Name: teamName, DisplayName: teamName, UserID: teamUserID}, nil)

		server = NewServer(&ServerParams{
			Cache:            cache,
			DBWriter:         dbWriter,
			DBReader:         dbReader,
			Pushy:            pushy,
			Users:            users,
			TelemetryHandler: &telemetryhook.NoopClient{},
		})
	})

	Context("when the request body is malformed", func() {
		It("fails with Bad Request when channel ID is invalid", func() {
			path := fmt.Sprintf("/v1/channels/%s/teams/%s/invitation", "123", "456")

			reqBody := `{
				"something": "else"
			}`

			buffer := bytes.NewBufferString(reqBody)

			req, err := http.NewRequest(http.MethodPut, path, buffer)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("accept status is required"))
		})

		It("fails with Bad Request when request body is invalid", func() {
			path := fmt.Sprintf("/v1/channels/%s/teams/%s/invitation", "123", "123")
			req, err := http.NewRequest(http.MethodPut, path, bytes.NewReader([]byte{}))
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)

			Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid request body"))
		})
	})

	It("fails with Bad Request when channel ID is invalid", func() {
		path := fmt.Sprintf("/v1/channels/%s/teams/%s/invitation", "=D", "123")

		reqBody := `{
			"accepted": true
		}`

		buffer := bytes.NewBufferString(reqBody)

		req, err := http.NewRequest(http.MethodPut, path, buffer)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
		Expect(recorder.Body.String()).To(ContainSubstring("invalid channel id (must be numeric)"))
	})

	It("fails with Bad Request when team ID is invalid", func() {
		path := fmt.Sprintf("/v1/channels/%s/teams/%s/invitation", "123", "=D")

		reqBody := `{
			"accepted": true
		}`

		buffer := bytes.NewBufferString(reqBody)

		req, err := http.NewRequest(http.MethodPut, path, buffer)
		Expect(err).NotTo(HaveOccurred())

		server.ServeHTTP(recorder, req)

		Expect(recorder.Result().StatusCode).To(Equal(http.StatusBadRequest))
		Expect(recorder.Body.String()).To(ContainSubstring("invalid team id (must be numeric)"))
	})

	Context("when the request parameters are valid", func() {
		JustBeforeEach(func() {
			path := fmt.Sprintf("/v1/channels/%s/teams/%s/invitation", channelID, teamID)
			reqBody := v1.PutChannelInvitationRequestBody{
				Accepted: &accepted,
			}

			buffer := new(bytes.Buffer)
			err := json.NewEncoder(buffer).Encode(&reqBody)
			Expect(err).NotTo(HaveOccurred())

			req, err := http.NewRequest(http.MethodPut, path, buffer)
			Expect(err).NotTo(HaveOccurred())

			server.ServeHTTP(recorder, req)
		})

		Context("when checking that the channel exists via User Service", func() {
			Context("when the User Service errors", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, errors.New("some error"))
				})

				It("fails with Internal Server Error", func() {
					users.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
					Expect(recorder.Body.String()).To(ContainSubstring("user service internal error"))
				})
			})

			Context("when the User Service does not find the channel/user", func() {
				BeforeEach(func() {
					users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, &channels.ErrChannelNotFound{})
				})

				It("fails with Not Found", func() {
					users.AssertExpectations(GinkgoT())

					Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
					Expect(recorder.Body.String()).To(ContainSubstring("channel with id 456 not found"))
				})
			})
		})

		Context("when executing the transaction via db client", func() {
			BeforeEach(func() {
				users.On("Get", mock.Anything, channelID, mock.Anything).Return(nil, nil)
			})

			Context("when accept field in request body is true", func() {
				Context("when AcceptInvitation has problems", func() {
					Context("when the membership fails to be created", func() {
						BeforeEach(func() {

							dbWriter.On("AcceptInvitation", mock.Anything, teamID, channelID).Return(db.ErrNoMembershipCreated)
						})

						It("returns with membership not created error", func() {
							dbWriter.AssertExpectations(GinkgoT())
							users.AssertExpectations(GinkgoT())

							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
							Expect(recorder.Body.String()).To(ContainSubstring("membership not created successfully"))
						})
					})

					Context("when the invitation does NOT exist", func() {
						BeforeEach(func() {
							dbWriter.On("AcceptInvitation", mock.Anything, teamID, channelID).Return(db.ErrNoRowFoundForDeletion)
						})

						It("fails with invitation not found", func() {
							dbWriter.AssertExpectations(GinkgoT())
							users.AssertExpectations(GinkgoT())

							Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
							Expect(recorder.Body.String()).To(ContainSubstring("invitation not found"))
						})
					})

					Context("when the AcceptInvitation internal errors", func() {
						BeforeEach(func() {
							dbWriter.On("AcceptInvitation", mock.Anything, teamID, channelID).Return(errors.New("some error"))
						})

						It("fails with Internal Server Error", func() {
							dbWriter.AssertExpectations(GinkgoT())
							users.AssertExpectations(GinkgoT())

							Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
							Expect(recorder.Body.String()).To(ContainSubstring("db internal error"))
						})
					})
				})

				Context("when AcceptInvitation is successful", func() {
					BeforeEach(func() {
						dbWriter.On("AcceptInvitation", mock.Anything, teamID, channelID).Return(nil)
						cache.On("ClearChannelMemberships", mock.Anything, channelID).Return(nil)
						cache.On("ClearAllTeamMembershipsForTeam", mock.Anything, teamID).Return(nil)
						pushy.On("PublishInviteAccepted", mock.Anything, teamUserID, teamName, channelID).Return(nil)
					})

					It("returns a 204", func() {
						dbWriter.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())

						Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
					})
				})
			})

			Context("when accept field in request body is false", func() {
				Context("when the invitation does NOT exist", func() {
					BeforeEach(func() {
						accepted = false
						dbWriter.On("DeleteInvitation", mock.Anything, teamID, channelID).Return(db.ErrNoRowFoundForDeletion)
						pushy.On("PublishInviteDeclined", mock.Anything, teamUserID, teamName, channelID).Return(nil)
					})

					It("fails with invitation not found", func() {
						dbWriter.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())
						Expect(recorder.Result().StatusCode).To(Equal(http.StatusNotFound))
						Expect(recorder.Body.String()).To(ContainSubstring("invitation not found"))
					})
				})

				Context("when the delete invitation internal errors", func() {
					BeforeEach(func() {
						accepted = false
						dbWriter.On("DeleteInvitation", mock.Anything, teamID, channelID).Return(errors.New("some error"))
					})

					It("fails with Internal Server Error", func() {
						dbWriter.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())

						Expect(recorder.Result().StatusCode).To(Equal(http.StatusInternalServerError))
						Expect(recorder.Body.String()).To(ContainSubstring("db internal error"))
					})
				})

				Context("when delete invitation is successful", func() {
					BeforeEach(func() {
						accepted = false
						dbWriter.On("DeleteInvitation", mock.Anything, teamID, channelID).Return(nil)
						pushy.On("PublishInviteDeclined", mock.Anything, teamUserID, teamName, channelID).Return(nil)
					})

					It("returns a 204", func() {
						dbWriter.AssertExpectations(GinkgoT())
						users.AssertExpectations(GinkgoT())

						Expect(recorder.Result().StatusCode).To(Equal(http.StatusNoContent))
					})
				})
			})

		})
	})
})
