package api

import (
	"context"
	"encoding/json"
	"net/http"
	"strconv"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"code.justin.tv/web/users-service/models"
)

// PUT /v1/channels/:channel_id/teams/:team_id/membership
func (s *Server) putV1ChannelMembership(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)
	ctx := req.Context()
	channelID := ctx.Value(contextKeyChannelID).(string)
	teamID := ctx.Value(contextKeyTeamID).(string)

	uint64ChannelID, err := strconv.ParseUint(channelID, 10, 64)
	if err != nil {
		jsonWriter.BadRequest("invalid channel id")
		return
	}

	uint64TeamID, err := strconv.ParseUint(teamID, 10, 64)
	if err != nil {
		jsonWriter.BadRequest("invalid team id")
		return
	}

	reqBody := v1.PutChannelMembershipRequestBody{}

	err = json.NewDecoder(req.Body).Decode(&reqBody)
	if err != nil {
		jsonWriter.BadRequest("invalid request body")
		return
	}

	if reqBody.RevenueRevealed == nil {
		jsonWriter.BadRequest("parameter 'revenue_revealed' is required")
		return
	}

	if reqBody.StatsRevealed == nil {
		jsonWriter.BadRequest("parameter 'stats_revealed' is required")
		return
	}

	err = s.dbWriter.UpdateMembership(ctx, teamID, channelID, *reqBody.RevenueRevealed, *reqBody.StatsRevealed)
	if err != nil {
		switch err {
		case db.ErrNoMembershipForUpdate:
			jsonWriter.NotFound("membership not found")
		default:
			jsonWriter.InternalServerError("db: failed to update membership", err)
		}
		return
	}

	if reqBody.Primary != nil && *reqBody.Primary {
		err = s.users.Set(ctx, models.UpdateChannelProperties{
			ID:            uint64ChannelID,
			PrimaryTeamID: &uint64TeamID,
		}, nil)

		if err != nil {
			jsonWriter.InternalServerError("users service: failed to update channel's primary team id", err)
			return
		}
	}

	go s.expireCachedChannelMemberships(context.Background(), channelID)
	go s.expireCachedTeamMemberships(context.Background(), teamID)

	w.WriteHeader(http.StatusNoContent)
}
