package api

import (
	"errors"
	"fmt"
	"net/http"
	"net/url"
	"strconv"

	"code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/db"
	"code.justin.tv/cb/roster/internal/httputil"
	"golang.org/x/sync/errgroup"
)

// GET /v1/search/teams?name=
func (s *Server) searchV1Teams(w http.ResponseWriter, req *http.Request) {
	jsonWriter := httputil.NewJSONResponseWriter(w)

	filter, err := toDBSearchTeamsFilter(req.URL.Query())
	if err != nil {
		jsonWriter.BadRequest(err.Error())
		return
	}

	group, ctx := errgroup.WithContext(req.Context())

	var total uint
	var teams []db.Team

	group.Go(func() error {
		var countError error
		total, countError = s.dbReader.SearchTeamsCount(ctx, filter)
		return countError
	})

	group.Go(func() error {
		var teamsError error
		teams, teamsError = s.dbReader.SearchTeams(ctx, filter)
		return teamsError
	})

	if err = group.Wait(); err != nil {
		jsonWriter.InternalServerError("failed to query teams from db", err)
		return
	}

	response := v1.SearchTeamsResponse{
		Meta: v1.SearchTeamsMeta{
			Limit:  filter.Limit,
			Offset: filter.Offset,
			Total:  total,
		},
		Data: toV1Teams(teams),
	}

	jsonWriter.OK(response)
}

func toDBSearchTeamsFilter(query url.Values) (db.SearchTeamsFilter, error) {
	limit := uint(db.GetTeamsDefaultLimit)
	offset := uint(0)

	name := query.Get("name")
	if name == "" {
		return db.SearchTeamsFilter{}, errors.New("name is required")
	}

	if limitQuery := query.Get("limit"); limitQuery != "" {
		parsed, err := strconv.ParseUint(limitQuery, 10, 64)
		if err != nil {
			return db.SearchTeamsFilter{}, errors.New("invalid limit")
		}

		if parsed > db.GetTeamsMaxLimit {
			return db.SearchTeamsFilter{}, fmt.Errorf("limit cannot be greater than %d", db.GetTeamsMaxLimit)
		}

		limit = uint(parsed)
	}

	if offsetQuery := query.Get("offset"); offsetQuery != "" {
		parsed, err := strconv.ParseUint(offsetQuery, 10, 64)
		if err != nil {
			return db.SearchTeamsFilter{}, errors.New("invalid offset")
		}

		offset = uint(parsed)
	}

	return db.SearchTeamsFilter{
		Limit:  limit,
		Name:   name,
		Offset: offset,
	}, nil
}
