package api

import (
	"encoding/json"
	"errors"
	"net/http"
	"net/http/httptest"
	"time"

	v1 "code.justin.tv/cb/roster/api/v1"
	"code.justin.tv/cb/roster/internal/api/mocks"
	"code.justin.tv/cb/roster/internal/clients/telemetryhook"
	"code.justin.tv/cb/roster/internal/db"
	. "github.com/onsi/ginkgo"
	. "github.com/onsi/gomega"
	"github.com/stretchr/testify/mock"
)

var _ = Describe("SearchV1Teams", func() {
	var (
		dbReader *mocks.DBReader
		server   *Server
		recorder *httptest.ResponseRecorder
		req      *http.Request
		err      error

		searchName, limit, offset string
	)

	BeforeEach(func() {
		recorder = httptest.NewRecorder()
		dbReader = &mocks.DBReader{}

		server = NewServer(&ServerParams{
			DBReader:         dbReader,
			Users:            &mocks.Users{},
			TelemetryHandler: &telemetryhook.NoopClient{},
		})

		searchName = "taf"
	})

	Context("with missing name", func() {
		BeforeEach(func() {
			searchName = ""

			req, err = http.NewRequest(http.MethodGet, "/v1/search/teams", nil)
			Expect(err).NotTo(HaveOccurred())

			query := req.URL.Query()
			query.Set("name", searchName)
			req.URL.RawQuery = query.Encode()
		})

		It("returns 400", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("name is required"))
		})
	})

	Context("with invalid limit", func() {
		BeforeEach(func() {
			limit = "abc"

			req, err = http.NewRequest(http.MethodGet, "/v1/search/teams", nil)
			Expect(err).NotTo(HaveOccurred())

			query := req.URL.Query()
			query.Set("name", searchName)
			query.Set("limit", limit)
			req.URL.RawQuery = query.Encode()
		})

		It("returns 400", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid limit"))
		})
	})

	Context("with limit too large", func() {
		BeforeEach(func() {
			limit = "101"

			req, err = http.NewRequest(http.MethodGet, "/v1/search/teams", nil)
			Expect(err).NotTo(HaveOccurred())

			query := req.URL.Query()
			query.Set("name", searchName)
			query.Set("limit", limit)
			req.URL.RawQuery = query.Encode()
		})

		It("returns 400", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("limit cannot be greater than"))
		})
	})

	Context("with invalid offset", func() {
		BeforeEach(func() {
			offset = "abc"

			req, err = http.NewRequest(http.MethodGet, "/v1/search/teams", nil)
			Expect(err).NotTo(HaveOccurred())

			query := req.URL.Query()
			query.Set("name", searchName)
			query.Set("offset", offset)
			req.URL.RawQuery = query.Encode()
		})

		It("returns 400", func() {
			server.ServeHTTP(recorder, req)

			Expect(recorder.Code).To(Equal(http.StatusBadRequest))
			Expect(recorder.Body.String()).To(ContainSubstring("invalid offset"))
		})
	})

	Context("with valid name, limit, and offset", func() {
		BeforeEach(func() {
			limit = "9"
			offset = "2"

			req, err = http.NewRequest(http.MethodGet, "/v1/search/teams", nil)
			Expect(err).NotTo(HaveOccurred())

			query := req.URL.Query()
			query.Set("name", searchName)
			query.Set("limit", limit)
			query.Set("offset", offset)
			req.URL.RawQuery = query.Encode()
		})

		Context("when DB fails to select the total team count", func() {
			It("Returns 500", func() {
				dbReader.On("SearchTeamsCount", mock.Anything, mock.Anything).Return(uint(0), errors.New("failed to query"))
				dbReader.On("SearchTeams", mock.Anything, mock.Anything).Return([]db.Team{}, nil)

				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("when DB fails to select teams", func() {
			It("Returns 500", func() {
				dbReader.On("SearchTeamsCount", mock.Anything, mock.Anything).Return(uint(333), nil)
				dbReader.On("SearchTeams", mock.Anything, mock.Anything).Return([]db.Team{}, errors.New("failed to query"))

				server.ServeHTTP(recorder, req)

				Expect(recorder.Code).To(Equal(http.StatusInternalServerError))
			})
		})

		Context("when DB succeeds", func() {
			teamID := "team_id"
			teamName := "staff"
			displayName := "Team 1!"
			userID := "user_id"
			description := "<b>Bold team</b>"
			logoID := "logouid"
			logoURL := "logo.src"
			bannerID := "banneruid"
			bannerURL := "banner.src"
			backgroundID := "backgrounduid"
			backgroundURL := "background.src"
			createdAt := time.Date(2000, 1, 1, 0, 0, 0, 0, time.UTC)
			updatedAt := time.Date(3000, 12, 12, 1, 2, 3, 4, time.UTC)

			BeforeEach(func() {
				team := db.Team{
					ID:          teamID,
					Name:        teamName,
					UserID:      userID,
					DisplayName: displayName,
					Logo: &db.Image{
						ID:  logoID,
						URL: logoURL,
					},
					Banner: &db.Image{
						ID:  bannerID,
						URL: bannerURL,
					},
					Background: &db.Image{
						ID:  backgroundID,
						URL: backgroundURL,
					},
					CreatedAt: &createdAt,
					UpdatedAt: &updatedAt,
				}
				team.SetDescriptionWithHTML(description)

				filter := db.SearchTeamsFilter{
					Limit:  9,
					Name:   searchName,
					Offset: 2,
				}

				dbReader.On("SearchTeams", mock.Anything, filter).Return([]db.Team{team}, nil)

				dbReader.On("SearchTeamsCount", mock.Anything, filter).Return(uint(333), nil)
			})

			It("Returns found team when it is found", func() {
				expectedTeamData := v1.Team{
					ID:                  teamID,
					Name:                teamName,
					UserID:              userID,
					DisplayName:         displayName,
					DescriptionHTML:     description,
					DescriptionMarkdown: "**Bold team**",
					LogoID:              &logoID,
					LogoURL:             &logoURL,
					BannerID:            &bannerID,
					BannerURL:           &bannerURL,
					BackgroundImageID:   &backgroundID,
					BackgroundImageURL:  &backgroundURL,
					CreatedAt:           &createdAt,
					UpdatedAt:           &updatedAt,
				}

				server.ServeHTTP(recorder, req)

				response := v1.SearchTeamsResponse{}
				err = json.Unmarshal(recorder.Body.Bytes(), &response)
				Expect(err).NotTo(HaveOccurred())

				Expect(recorder.Code).To(Equal(http.StatusOK))
				Expect(response.Data).To(HaveLen(1))
				Expect(response.Data[0]).To(Equal(expectedTeamData))
			})
		})
	})
})
